/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2015-2019 Ping Identity Corporation
 */

package com.unboundid.directory.sdk.broker.types;

import com.unboundid.util.NotExtensible;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;



/**
 * An attribute in a Store Adapter's native schema. Use
 * StoreAttributeDefinition.Builder to create instances of this class.
 */
@NotExtensible()
public final class StoreAttributeDefinition
{
  /**
   * An enumeration of the data types for values.
   */
  public enum Type
  {
    /**
     * String datatype.
     */
    STRING("string"),

    /**
     * Boolean datatype.
     */
    BOOLEAN("boolean"),

    /**
     * Decimal datatype.
     */
    DECIMAL("decimal"),

    /**
     * Integer datatype.
     */
    INTEGER("integer"),

    /**
     * Datetime datatype.
     */
    DATETIME("datetime"),

    /**
     * Binary datatype.
     */
    BINARY("binary"),

    /**
     * Reference datatype.
     */
    REFERENCE("reference"),

    /**
     * Complex datatype.
     */
    COMPLEX("complex");

    private String name;

    /**
     * Constructs an attribute data type object.
     * @param name the name of the data type.
     */
    Type(final String name)
    {
      this.name = name;
    }

    /**
     * Gets the name of the type.
     *
     * @return the name of the type.
     */
    public String getName()
    {
      return name;
    }

    /**
     * Gets the Type matching the given name.  Throws a runtime
     * exception if the Type cannot be found because an invalid
     * name was given.
     *
     * @param name the name of the type.
     * @return the type matching the given name.
     */
    public static Type fromName(final String name)
    {
      for(Type type : Type.values())
      {
        if(type.getName().equals(name))
        {
          return type;
        }
      }

      throw new RuntimeException("Unknown StoreAttributeDefinition datatype");
    }
  }

  private final String name;
  private final Type type;
  private final String description;
  private final Collection<StoreAttributeDefinition> subAttributes;
  private final boolean multiValued;

  /**
   * Builder class to build an instance of StoreAttributeDefinition.
   */
  public static class Builder
  {
    /**
     * The name of the attribute.
     */
    private String name;

    /**
     * The type of the attribute.  For the possible values, see:
     * {@link StoreAttributeDefinition.Type}
     */
    private Type type;

    /**
     * The sub-attributes of a complex attribute.
     */
    private Collection<StoreAttributeDefinition> subAttributes;

    /**
     * A boolean value indicating whether or not this attribute can have
     * multiple values.
     */
    private boolean multiValued;

    /**
     * The description of this attribute.
     */
    private String description;

    /**
     * Create a new builder.
     */
    public Builder()
    {
      type = Type.STRING;
    }

    /**
     * Sets the attribute name.
     *
     * @param name the attribute name.
     * @return this
     */
    public Builder setName(final String name)
    {
      this.name = name;
      return this;
    }

    /**
     * Sets the type of the attribute.
     *
     * @param type the type of the attribute.
     * @return this.
     */
    public Builder setType(final Type type)
    {
      this.type = type;
      return this;
    }

    /**
     * Adds sub-attributes for a complex datatype attribute.
     *
     * @param subAttributes  The sub-attributes of the attribute.
     * @return this.
     */
    public Builder addSubAttributes(
        final StoreAttributeDefinition ... subAttributes)
    {
      if (subAttributes != null && subAttributes.length > 0)
      {
        if (this.subAttributes == null)
        {
          this.subAttributes = new LinkedList<StoreAttributeDefinition>();
        }
        this.subAttributes.addAll(Arrays.asList(subAttributes));
      }
      return this;
    }

    /**
     * Sets a boolean indicating if the attribute is multi-valued.
     *
     * @param multiValued a boolean indicating if the attribute is multi-valued.
     * @return this.
     */
    public Builder setMultiValued(final boolean multiValued)
    {
      this.multiValued = multiValued;
      return this;
    }

    /**
     * Sets the description of the attribute.
     *
     * @param description the description of the attribute.
     * @return this.
     */
    public Builder setDescription(final String description)
    {
      this.description = description;
      return this;
    }

    /**
     * Clears all values in this builder, so that it can be reused.
     *
     * @return this.
     */
    public Builder clear()
    {
      this.name = null;
      this.type = Type.STRING;
      this.subAttributes = null;
      this.multiValued = false;
      this.description = null;
      return this;
    }

    /**
     * Builds a new StoreAttributeDefinition.
     *
     * @return a new StoreAttributeDefinition.
     */
    public StoreAttributeDefinition build()
    {
      return new StoreAttributeDefinition(
          name,
          type,
          subAttributes,
          multiValued,
          description);
    }
  }

  /**
   * Create a new Attribute Definition.
   *
   * @param name The name of the attribute.
   * @param type The attribute data type.
   * @param subAttributes The sub-attributes of the attribute.
   * @param multiValued A boolean indicating if the attribute is multi-valued.
   * @param description The description of this attribute.
   */
  private StoreAttributeDefinition(
      final String name,
      final Type type,
      final Collection<StoreAttributeDefinition> subAttributes,
      final boolean multiValued,
      final String description)
  {
    this.name = name;
    this.type = type;
    this.subAttributes = subAttributes == null ?
        null : Collections.unmodifiableList(
        new ArrayList<StoreAttributeDefinition>(subAttributes));
    this.multiValued = multiValued;
    this.description = description;
  }

  /**
   * Determines if the attribute allows multiple values.
   *
   * @return true if the attribute is multivalued, or false if it is not.
   */
  public boolean isMultiValued()
  {
    return multiValued;
  }

  /**
   * Gets the type of the value for this attribute.
   *
   * @return type of the value for this attribute.
   */
  public Type getType()
  {
    return type;
  }

  /**
   * Gets the name of the attribute.
   *
   * @return the name of the attribute.
   */
  public String getName()
  {
    return name;
  }

  /**
   * Gets the description of the attribute.
   *
   * @return the description of the attribute.
   */
  public String getDescription()
  {
    return description;
  }

  /**
   * Gets the sub-attributes for a complex attribute.
   *
   * @return the sub-attributes for a complex attribute.
   */
  public Collection<StoreAttributeDefinition> getSubAttributes()
  {
    return subAttributes;
  }

  /**
   * Gets a string representation of the attribute.
   *
   * @return a string representation of the attribute.
   */
  @Override
  public String toString()
  {
    return toIndentedString("");
  }

  /**
   * Called by toString.  This is used to format the output of the object
   * a little to help readability.
   *
   * @param indent the string to use for each indent increment.  For example,
   *               one might use "  " for a 2 space indent.
   * @return a string representation of this attribute.
   */
  private String toIndentedString(final String indent)
  {
    StringBuilder builder = new StringBuilder();
    builder.append(indent);
    builder.append("Name: ");
    builder.append(getName());
    builder.append(" Description: ");
    builder.append(getDescription());
    builder.append(" isReadOnly: ");
    builder.append(System.lineSeparator());
    if(getSubAttributes() != null)
    {
      for (StoreAttributeDefinition a : getSubAttributes())
      {
        builder.append(a.toIndentedString(indent + "  "));
      }
    }
    return builder.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(final Object o)
  {
    if (this == o)
    {
      return true;
    }
    if (o == null || getClass() != o.getClass())
    {
      return false;
    }

    StoreAttributeDefinition that = (StoreAttributeDefinition) o;

    if (multiValued != that.multiValued)
    {
      return false;
    }
    if (description != null ? !description.equals(that.description) :
        that.description != null)
    {
      return false;
    }
    if (name != null ? !name.equals(that.name) : that.name != null)
    {
      return false;
    }
    if (subAttributes != null ? !subAttributes.equals(that.subAttributes) :
        that.subAttributes != null)
    {
      return false;
    }
    if (type != null ? !type.equals(that.type) : that.type != null)
    {
      return false;
    }

    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode()
  {
    int result = name != null ? name.hashCode() : 0;
    result = 31 * result + (type != null ? type.hashCode() : 0);
    result = 31 * result + (subAttributes != null ?
        subAttributes.hashCode() : 0);
    result = 31 * result + (multiValued ? 1 : 0);
    result = 31 * result + (description != null ? description.hashCode() : 0);
    return result;
  }
}
