/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2013-2019 Ping Identity Corporation
 */

package com.unboundid.directory.sdk.broker.types;

import com.unboundid.util.NotExtensible;



/**
 * Represents a request to update a specific entry in a Store Adapter.
 *
 * The UnboundID SCIM 2 SDK can be used to parse a SCIM patch request:
 * <BR><BR>
 * <PRE>
 *   final ObjectReader reader =
 *     JsonUtils.getObjectReader().forType(PatchRequest.class);
 *   final PatchRequest patch =
 *     reader.readValue(request.getPatchRequest());
 * </PRE>
 * The easiest way to process the update request is to retrieve the entry and
 * apply the patch in memory using the PatchOperation.apply method using the
 * SCIM 2 SDK:
 * <BR><BR>
 * <PRE>
 *   // Retrieve the current entry from the store
 *   final GenericScimResource resourceToUpdate = read(...);
 *
 *   // Apply the patch request in memory
 *   patchOp.apply(resourceToUpdate);
 *
 *   // Replace the entire entry in the store
 *   replace(resourceToUpdate);
 * </PRE>
 * Alternatively, if the underlying store does not support replacing the entire
 * entry or to avoid the initial retrieval required to apply the patch in
 * memory, the individual patch operations may be mapped to update requests
 * supported by the underlying store. SCIM 2 (RFC 7644) patch operations are
 * quite complicated in that an operation's path and value may take many forms
 * but accomplish the same result. To make implementation easier, the patch
 * operations in this request have been normalized so that there is only one
 * form to accomplish a particular result. The operations in this request will
 * have one of the following forms depending on the operation type, whether the
 * attribute type is complex, and whether it is multi-valued:
 *
 * <H2>Add Operation</H2>
 * <TABLE BORDER="1" SUMMARY="Patch operation forms for add">
 *   <TR>
 *     <TH>
 *       "op" field
 *     </TH>
 *     <TH>
 *       "path" field
 *     </TH>
 *     <TH>
 *       "value" field
 *     </TH>
 *     <TH>
 *       Description
 *     </TH>
 *   </TR>
 *   <TR>
 *    <TD>add</TD>
 *    <TD>attributeName</TD>
 *    <TD>valueNode or objectNode</TD>
 *    <TD>
 *      <UL>
 *        <LI>
 *          If the attribute does not exist, the attribute and value are added
 *          to the resource.
 *        </LI>
 *        <LI>
 *          If the attribute exists in the resource, the existing value is
 *          replaced. If the existing value and the "value" parameter are
 *          objects containing sub-attributes, those that did not previously
 *          exist are added or their value replaced if they already exist.
 *          Sub-attributes that are not specified in the "value" parameter are
 *          left unchanged.
 *        </LI>
 *      </UL>
 *    </TD>
 *   </TR>
 *   <TR>
 *    <TD>add</TD>
 *    <TD>attributeName</TD>
 *    <TD>[valueNodes] or [objectNodes]</TD>
 *    <TD>
 *      <UL>
 *        <LI>
 *          If the attribute does not exist, the attribute and values are added
 *          to the resource.
 *        </LI>
 *        <LI>
 *          If the attribute exists in the resource, the new values are added
 *          to the existing values.
 *        </LI>
 *      </UL>
 *    </TD>
 *   </TR>
 * </TABLE>
 * <H2>Remove Operation</H2>
 * <TABLE BORDER="1" SUMMARY="Patch operation forms for remove">
 *   <TR>
 *     <TH>
 *       "op" field
 *     </TH>
 *     <TH>
 *       "path" field
 *     </TH>
 *     <TH>
 *       "value" field
 *     </TH>
 *     <TH>
 *       Description
 *     </TH>
 *   </TR>
 *   <TR>
 *    <TD>remove</TD>
 *    <TD>attributeName</TD>
 *    <TD></TD>
 *    <TD>
 *      <UL>
 *        <LI>
 *          Remove the attribute and its associated value(s) from the resource.
 *        </LI>
 *      </UL>
 *    </TD>
 *   </TR>
 *   <TR>
 *    <TD>remove</TD>
 *    <TD>attributeName.subAttributeName</TD>
 *    <TD></TD>
 *    <TD>
 *      <UL>
 *        <LI>
 *          Remove the sub-attribute from the attribute's associated value(s).
 *          If no other sub-attributes remain after removal of the
 *          targeted sub-attribute, the attribute should be removed from the
 *          resource.
 *        </LI>
 *      </UL>
 *    </TD>
 *   </TR>
 *   <TR>
 *    <TD>remove</TD>
 *    <TD>attributeName[valueFilter]</TD>
 *    <TD></TD>
 *    <TD>
 *      <UL>
 *        <LI>
 *          Remove only the value(s) of the multi-valued attribute that matches
 *          the value filter. For simple multi-valued attributes, the value
 *          filter will be on an implicit "value" sub-attribute
 *          (ie. value eq 100). If no other values remain after removal of the
 *          selected values, the attribute should be removed from the resource.
 *        </LI>
 *      </UL>
 *    </TD>
 *   </TR>
 *   <TR>
 *    <TD>remove</TD>
 *    <TD>attributeName[valueFilter].subAttributeName</TD>
 *    <TD></TD>
 *    <TD>
 *      <UL>
 *        <LI>
 *          Remove the sub-attribute from any values of the multi-valued complex
 *          attribute that matches the value filter. If no other values remain
 *          after removal of the selected values, the attribute should be
 *          removed from the resource.
 *        </LI>
 *      </UL>
 *    </TD>
 *   </TR>
 * </TABLE>
 * <H2>Replace Operation</H2>
 * <TABLE BORDER="1" SUMMARY="Patch operation forms for replace">
 *   <TR>
 *     <TH>
 *       "op" field
 *     </TH>
 *     <TH>
 *       "path" field
 *     </TH>
 *     <TH>
 *       "value" field
 *     </TH>
 *     <TH>
 *       Description
 *     </TH>
 *   </TR>
 *   <TR>
 *    <TD>replace</TD>
 *    <TD>attributeName</TD>
 *    <TD>valueNode or objectNode</TD>
 *    <TD>
 *      <UL>
 *        <LI>
 *          If the attribute does not exist, the attribute and value are added
 *          to the resource. (Same as add operation)
 *        </LI>
 *        <LI>
 *          If the attribute exists in the resource, the existing value is
 *          replaced. If the existing value and the "value" parameter are
 *          objects containing sub-attributes, those that did not previously
 *          exist are added or their value replaced if they already exist.
 *          Sub-attributes that are not specified in the "value" parameter are
 *          left unchanged. (Same as add operation)
 *        </LI>
 *      </UL>
 *    </TD>
 *   </TR>
 *   <TR>
 *    <TD>replace</TD>
 *    <TD>attributeName</TD>
 *    <TD>[valueNodes] or [objectNodes]</TD>
 *    <TD>
 *      <UL>
 *        <LI>
 *          If the attribute does not exist, the attribute and values are added
 *          to the resource.
 *        </LI>
 *        <LI>
 *          If the attribute exists in the resource, existing values are
 *          replaced by new values.
 *        </LI>
 *      </UL>
 *    </TD>
 *   </TR>
 *   <TR>
 *    <TD>replace</TD>
 *    <TD>attributeName[comparisonFilter]</TD>
 *    <TD>valueNode or objectNode</TD>
 *    <TD>
 *      <UL>
 *        <LI>
 *          Replace only the value(s) of the multi-valued attribute that matches
 *          the value filter. For simple multi-valued attributes, the value
 *          filter will be on an implicit "value" sub-attribute
 *          (ie. value eq 100). If no other values remain after removal of the
 *          selected values, the attribute should be removed from the resource.
 *          If the matched value and the "value" parameter are
 *          objects containing sub-attributes, those that did not previously
 *          exist are added or their value replaced if they already exists.
 *          Sub-attributes that are not specified in the "value" parameter are
 *          left unchanged.
 *        </LI>
 *      </UL>
 *    </TD>
 * </TABLE>
 */
@NotExtensible
public interface StoreUpdateRequest
    extends StoreRequest
{
  /**
   * Retrieve the filter which identifies the entry to be updated.
   * The UnboundID scim2 SDK Filter class can be used to parse a SCIM filter.
   *
   * @return  The filter which identifies the entry to be updated.
   */
  String getSCIMFilter();

  /**
   * Retrieve the SCIM patch request specifying the modifications to be
   * processed.
   *
   * @return  The SCIM patch request specifying the modifications to be
   * processed.
   */
  String getPatchRequest();
}
