/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.api;



import java.security.cert.Certificate;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.unboundid.directory.sdk.broker.internal.BrokerExtension;
import com.unboundid.directory.sdk.common.config.AccessLoggerConfig;
import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.operation.AbandonRequest;
import com.unboundid.directory.sdk.common.operation.AddRequest;
import com.unboundid.directory.sdk.common.operation.AddResult;
import com.unboundid.directory.sdk.common.operation.BindResult;
import com.unboundid.directory.sdk.common.operation.CompareRequest;
import com.unboundid.directory.sdk.common.operation.CompareResult;
import com.unboundid.directory.sdk.common.operation.DeleteRequest;
import com.unboundid.directory.sdk.common.operation.DeleteResult;
import com.unboundid.directory.sdk.common.operation.ExtendedRequest;
import com.unboundid.directory.sdk.common.operation.ExtendedResult;
import com.unboundid.directory.sdk.common.operation.GenericResult;
import com.unboundid.directory.sdk.common.operation.ModifyRequest;
import com.unboundid.directory.sdk.common.operation.ModifyResult;
import com.unboundid.directory.sdk.common.operation.ModifyDNRequest;
import com.unboundid.directory.sdk.common.operation.ModifyDNResult;
import com.unboundid.directory.sdk.common.operation.SASLBindRequest;
import com.unboundid.directory.sdk.common.operation.SearchRequest;
import com.unboundid.directory.sdk.common.operation.SearchResult;
import com.unboundid.directory.sdk.common.operation.SimpleBindRequest;
import com.unboundid.directory.sdk.common.operation.UnbindRequest;
import com.unboundid.directory.sdk.common.types.AssuredReplicationResult;
import com.unboundid.directory.sdk.common.types.ClientContext;
import com.unboundid.directory.sdk.common.types.CompletedOperationContext;
import com.unboundid.directory.sdk.common.types.CompletedSearchOperationContext;
import com.unboundid.directory.sdk.common.types.DisconnectReason;
import com.unboundid.directory.sdk.common.types.Entry;
import com.unboundid.directory.sdk.common.types.ForwardTarget;
import com.unboundid.directory.sdk.common.types.OperationContext;
import com.unboundid.directory.sdk.common.types.ServerContext;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.metrics.internal.MetricsEngineExtension;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.Control;
import com.unboundid.ldap.sdk.IntermediateResponse;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.ldap.sdk.unboundidds.MoveSubtreeResult;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which
 * record information about interaction with clients, including connections
 * established and received and operations requested and completed.  Access
 * loggers may write information to files, but they may also write to other
 * locations, including databases, message, queues, e-mail messages, or other
 * targets.
 * <BR><BR>
 * Access loggers will be invoked for the following events:
 * <UL>
 *   <LI>Whenever a new connection is established.</LI>
 *   <LI>Whenever an existing connection is closed or terminated.</LI>
 *   <LI>Whenever an abandon, add, bind, compare, delete, extended, modify,
 *       modify DN, search, or unbind request is received.</LI>
 *   <LI>Whenever an abandon, add, bind, compare, delete, extended, modify,
 *       modify DN, or search request is forwarded to another server for
 *       processing.</LI>
 *   <LI>If a forwarded add, bind, compare, delete, extended, modify, modify DN,
 *       or search operation fails.</LI>
 *   <LI>After sending the result for an add, bind, compare, delete, extended,
 *       modify, modify DN, or search operation.</LI>
 *   <LI>After completing processing for an abandon operation.</LI>
 *   <LI>After sending a search result entry, search result reference, or
 *       intermediate response message to the client.</LI>
 * </UL>
 * <BR><BR>
 * Each access logger may configured to indicate whether to include or exclude
 * internal and/or replicated operations, and criteria may be used to provide
 * filtered logging.  This is handled automatically by the server, so individual
 * access logger implementations do not need to attempt to perform that
 * filtering on their own.  However, they may perform additional processing if
 * desired to further narrow the set of messages that should be logged.
 * <BR>
 * <H2>Configuring Access Loggers</H2>
 * In order to configure an access logger created using this API, use a command
 * like:
 * <PRE>
 *      dsconfig create-log-publisher \
 *           --publisher-name "<I>{logger-name}</I>" \
 *           --type third-party-access \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{logger-name}</I>" is the name to use for the access logger
 * instance, "<I>{class-name}</I>" is the fully-qualified name of the Java class
 * that extends {@code com.unboundid.directory.sdk.common.api.AccessLogger},
 * and "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the logger.  If multiple arguments should be provided to the
 * logger, then the "<CODE>--set extension-argument:<I>{name=value}</I></CODE>"
 * option should be provided multiple times.
 *
 * @see  FileBasedAccessLogger
 * @see  com.unboundid.directory.sdk.common.scripting.ScriptedAccessLogger
 * @see
 *    com.unboundid.directory.sdk.common.scripting.ScriptedFileBasedAccessLogger
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=true)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@MetricsEngineExtension()
@BrokerExtension()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class AccessLogger
       implements UnboundIDExtension, Reconfigurable<AccessLoggerConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this access logger.  All access logger
   * implementations must include a default constructor, but any initialization
   * should generally be done in the {@code initializeAccessLogger} method.
   */
  public AccessLogger()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this access logger.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this access logger.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this access logger.
   *
   * @throws  LDAPException  If a problem occurs while initializing this access
   *                         logger.
   */
  public void initializeAccessLogger(final ServerContext serverContext,
                                     final AccessLoggerConfig config,
                                     final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(final AccessLoggerConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final AccessLoggerConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this access logger " +
                "is disabled and re-enabled or until the server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this access logger is to
   * be taken out of service.
   */
  public void finalizeAccessLogger()
  {
    // No implementation is required.
  }



  /**
   * Logs a message indicating that a new connection has been established.
   *
   * @param  clientContext  Information about the client connection that has
   *                        been accepted.
   */
  public void logConnect(final ClientContext clientContext)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message indicating that a connection has been closed.
   *
   * @param  clientContext     Information about the client connection that has
   *                           been closed.
   * @param  disconnectReason  A general reason that the connection has been
   *                           closed.
   * @param  message           A message with additional information about the
   *                           closure.  It may be {@code null} if none is
   *                           available.
   */
  public void logDisconnect(final ClientContext clientContext,
                            final DisconnectReason disconnectReason,
                            final String message)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about security negotiation performed by a client.
   *
   * @param  clientContext  Information about the client connection on which
   *                        the negotiation was completed.
   * @param  protocol       The security protocol selected by the negotiation.
   *                        It may be {@code null} if no protocol is available.
   * @param  cipher         The cipher suite selected by the negotiation.  It
   *                        may be {@code null} if no cipher is available.
   * @param  properties     A set of additional properties that may be included
   *                        in the log message.  It may be {@code null} or empty
   *                        if no additional properties are needed.
   */
  public void logSecurityNegotiation(final ClientContext clientContext,
                                     final String protocol, final String cipher,
                                     final Map<String,String> properties)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a certificate chain presented by a client.
   *
   * @param  clientContext  Information about the client that presented the
   *                        certificate chain.
   * @param  certChain      The certificate chain presented by the client.
   * @param  authDN         The DN of the user as whom the client was
   *                        automatically authenticated, or {@code null} if the
   *                        client was not automatically authenticated.
   */
  public void logClientCertificateChain(final ClientContext clientContext,
                                        final Certificate[] certChain,
                                        final String authDN)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about an abandon request received from a client.
   *
   * @param  opContext  The operation context for the abandon operation.
   * @param  request    The abandon request that was received.
   */
  public void logAbandonRequest(final OperationContext opContext,
                                final AbandonRequest request)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about an abandon request that will be forwarded to another
   * server.
   *
   * @param  opContext  The operation context for the abandon operation.
   * @param  request    The abandon request that was received.
   * @param  target     Information about the server to which the request will
   *                    be forwarded.
   */
  public void logAbandonForward(final OperationContext opContext,
                                final AbandonRequest request,
                                final ForwardTarget target)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about the result of processing an abandon request.
   *
   * @param  opContext  The operation context for the abandon operation.
   * @param  request    The abandon request that was received.
   * @param  result     The result of processing the abandon request.
   */
  public void logAbandonResult(final CompletedOperationContext opContext,
                               final AbandonRequest request,
                               final GenericResult result)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about an add request received from a client.
   *
   * @param  opContext  The operation context for the add operation.
   * @param  request    The add request that was received.
   */
  public void logAddRequest(final OperationContext opContext,
                            final AddRequest request)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about an add request that will be forwarded to another
   * server.
   *
   * @param  opContext  The operation context for the add operation.
   * @param  request    The add request that was received.
   * @param  target     Information about the server to which the request will
   *                    be forwarded.
   */
  public void logAddForward(final OperationContext opContext,
                            final AddRequest request,
                            final ForwardTarget target)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a failure encountered while attempting to forward an
   * add request to another server.
   *
   * @param  opContext  The operation context for the add operation.
   * @param  request    The add request that was received.
   * @param  target     Information about the server to which the request was
   *                    forwarded.
   * @param  failure    The exception that was received when attempting to
   *                    forward the request.
   */
  public void logAddForwardFailure(final OperationContext opContext,
                                   final AddRequest request,
                                   final ForwardTarget target,
                                   final LDAPException failure)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about the result of processing an add request.
   *
   * @param  opContext  The operation context for the add operation.
   * @param  request    The add request that was received.
   * @param  result     The result of processing the add request.
   */
  public void logAddResponse(final CompletedOperationContext opContext,
                             final AddRequest request,
                             final AddResult result)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about the result of replication assurance processing for an
   * add operation.
   *
   * @param  opContext        The operation context for the add operation.
   * @param  request          The add request that was received.
   * @param  result           The result of processing the add request.
   * @param  assuranceResult  The replication assurance processing result.
   */
  public void logAddAssuranceCompleted(
                   final CompletedOperationContext opContext,
                   final AddRequest request, final AddResult result,
                   final AssuredReplicationResult assuranceResult)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a simple bind request received from a client.
   *
   * @param  opContext  The operation context for the bind operation.
   * @param  request    The bind request that was received.
   */
  public void logBindRequest(final OperationContext opContext,
                             final SimpleBindRequest request)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a simple bind request that will be forwarded to
   * another server.
   *
   * @param  opContext  The operation context for the bind operation.
   * @param  request    The bind request that was received.
   * @param  target     Information about the server to which the request will
   *                    be forwarded.
   */
  public void logBindForward(final OperationContext opContext,
                                final SimpleBindRequest request,
                                final ForwardTarget target)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a failure encountered while attempting to forward a
   * simple bind request to another server.
   *
   * @param  opContext  The operation context for the bind operation.
   * @param  request    The bind request that was received.
   * @param  target     Information about the server to which the request was
   *                    forwarded.
   * @param  failure    The exception that was received when attempting to
   *                    forward the request.
   */
  public void logBindForwardFailure(final OperationContext opContext,
                                    final SimpleBindRequest request,
                                    final ForwardTarget target,
                                    final LDAPException failure)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about the result of processing a simple bind request.
   *
   * @param  opContext  The operation context for the bind operation.
   * @param  request    The bind request that was received.
   * @param  result     The result of processing the bind request.
   */
  public void logBindResponse(final CompletedOperationContext opContext,
                              final SimpleBindRequest request,
                              final BindResult result)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a SASL bind request received from a client.
   *
   * @param  opContext  The operation context for the bind operation.
   * @param  request    The bind request that was received.
   */
  public void logBindRequest(final OperationContext opContext,
                             final SASLBindRequest request)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a SASL bind request that will be forwarded to
   * another server.
   *
   * @param  opContext  The operation context for the bind operation.
   * @param  request    The bind request that was received.
   * @param  target     Information about the server to which the request will
   *                    be forwarded.
   */
  public void logBindForward(final OperationContext opContext,
                                final SASLBindRequest request,
                                final ForwardTarget target)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a failure encountered while attempting to forward a
   * SASL bind request to another server.
   *
   * @param  opContext  The operation context for the bind operation.
   * @param  request    The bind request that was received.
   * @param  target     Information about the server to which the request was
   *                    forwarded.
   * @param  failure    The exception that was received when attempting to
   *                    forward the request.
   */
  public void logBindForwardFailure(final OperationContext opContext,
                                    final SASLBindRequest request,
                                    final ForwardTarget target,
                                    final LDAPException failure)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about the result of processing a SASL bind request.
   *
   * @param  opContext  The operation context for the bind operation.
   * @param  request    The bind request that was received.
   * @param  result     The result of processing the bind request.
   */
  public void logBindResponse(final CompletedOperationContext opContext,
                              final SASLBindRequest request,
                              final BindResult result)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a compare request received from a client.
   *
   * @param  opContext  The operation context for the compare operation.
   * @param  request    The compare request that was received.
   */
  public void logCompareRequest(final OperationContext opContext,
                                final CompareRequest request)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a compare request that will be forwarded to another
   * server.
   *
   * @param  opContext  The operation context for the compare operation.
   * @param  request    The compare request that was received.
   * @param  target     Information about the server to which the request will
   *                    be forwarded.
   */
  public void logCompareForward(final OperationContext opContext,
                                final CompareRequest request,
                                final ForwardTarget target)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a failure encountered while attempting to forward a
   * compare request to another server.
   *
   * @param  opContext  The operation context for the compare operation.
   * @param  request    The compare request that was received.
   * @param  target     Information about the server to which the request was
   *                    forwarded.
   * @param  failure    The exception that was received when attempting to
   *                    forward the request.
   */
  public void logCompareForwardFailure(final OperationContext opContext,
                                       final CompareRequest request,
                                       final ForwardTarget target,
                                       final LDAPException failure)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about the result of processing a compare request.
   *
   * @param  opContext  The operation context for the compare operation.
   * @param  request    The compare request that was received.
   * @param  result     The result of processing the compare request.
   */
  public void logCompareResponse(final CompletedOperationContext opContext,
                                 final CompareRequest request,
                                 final CompareResult result)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a delete request received from a client.
   *
   * @param  opContext  The operation context for the delete operation.
   * @param  request    The delete request that was received.
   */
  public void logDeleteRequest(final OperationContext opContext,
                               final DeleteRequest request)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a delete request that will be forwarded to another
   * server.
   *
   * @param  opContext  The operation context for the delete operation.
   * @param  request    The delete request that was received.
   * @param  target     Information about the server to which the request will
   *                    be forwarded.
   */
  public void logDeleteForward(final OperationContext opContext,
                               final DeleteRequest request,
                               final ForwardTarget target)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a failure encountered while attempting to forward a
   * delete request to another server.
   *
   * @param  opContext  The operation context for the delete operation.
   * @param  request    The delete request that was received.
   * @param  target     Information about the server to which the request was
   *                    forwarded.
   * @param  failure    The exception that was received when attempting to
   *                    forward the request.
   */
  public void logDeleteForwardFailure(final OperationContext opContext,
                                      final DeleteRequest request,
                                      final ForwardTarget target,
                                      final LDAPException failure)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about the result of processing a delete request.
   *
   * @param  opContext  The operation context for the delete operation.
   * @param  request    The delete request that was received.
   * @param  result     The result of processing the delete request.
   */
  public void logDeleteResponse(final CompletedOperationContext opContext,
                                final DeleteRequest request,
                                final DeleteResult result)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about the result of replication assurance processing for a
   * delete operation.
   *
   * @param  opContext        The operation context for the delete operation.
   * @param  request          The delete request that was received.
   * @param  result           The result of processing the delete request.
   * @param  assuranceResult  The replication assurance processing result.
   */
  public void logDeleteAssuranceCompleted(
                   final CompletedOperationContext opContext,
                   final DeleteRequest request, final DeleteResult result,
                   final AssuredReplicationResult assuranceResult)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about an extended request received from a client.
   *
   * @param  opContext  The operation context for the extended operation.
   * @param  request    The extended request that was received.
   */
  public void logExtendedRequest(final OperationContext opContext,
                                 final ExtendedRequest request)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about an extended request that will be forwarded to another
   * server.
   *
   * @param  opContext  The operation context for the extended operation.
   * @param  request    The extended request that was received.
   * @param  target     Information about the server to which the request will
   *                    be forwarded.
   */
  public void logExtendedForward(final OperationContext opContext,
                                 final ExtendedRequest request,
                                 final ForwardTarget target)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a failure encountered while attempting to forward an
   * extended request to another server.
   *
   * @param  opContext  The operation context for the extended operation.
   * @param  request    The extended request that was received.
   * @param  target     Information about the server to which the request was
   *                    forwarded.
   * @param  failure    The exception that was received when attempting to
   *                    forward the request.
   */
  public void logExtendedForwardFailure(final OperationContext opContext,
                                        final ExtendedRequest request,
                                        final ForwardTarget target,
                                        final LDAPException failure)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about the result of processing an extended request.
   *
   * @param  opContext  The operation context for the extended operation.
   * @param  request    The extended request that was received.
   * @param  result     The result of processing the extended request.
   */
  public void logExtendedResponse(final CompletedOperationContext opContext,
                                  final ExtendedRequest request,
                                  final ExtendedResult result)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a modify request received from a client.
   *
   * @param  opContext  The operation context for the modify operation.
   * @param  request    The modify request that was received.
   */
  public void logModifyRequest(final OperationContext opContext,
                               final ModifyRequest request)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a modify request that will be forwarded to another
   * server.
   *
   * @param  opContext  The operation context for the modify operation.
   * @param  request    The modify request that was received.
   * @param  target     Information about the server to which the request will
   *                    be forwarded.
   */
  public void logModifyForward(final OperationContext opContext,
                               final ModifyRequest request,
                               final ForwardTarget target)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a failure encountered while attempting to forward a
   * modify request to another server.
   *
   * @param  opContext  The operation context for the modify operation.
   * @param  request    The modify request that was received.
   * @param  target     Information about the server to which the request was
   *                    forwarded.
   * @param  failure    The exception that was received when attempting to
   *                    forward the request.
   */
  public void logModifyForwardFailure(final OperationContext opContext,
                                      final ModifyRequest request,
                                      final ForwardTarget target,
                                      final LDAPException failure)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about the result of processing a modify request.
   *
   * @param  opContext  The operation context for the modify operation.
   * @param  request    The modify request that was received.
   * @param  result     The result of processing the modify request.
   */
  public void logModifyResponse(final CompletedOperationContext opContext,
                                final ModifyRequest request,
                                final ModifyResult result)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about the result of replication assurance processing for a
   * modify operation.
   *
   * @param  opContext        The operation context for the modify operation.
   * @param  request          The modify request that was received.
   * @param  result           The result of processing the modify request.
   * @param  assuranceResult  The replication assurance processing result.
   */
  public void logModifyAssuranceCompleted(
                   final CompletedOperationContext opContext,
                   final ModifyRequest request, final ModifyResult result,
                   final AssuredReplicationResult assuranceResult)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a modify DN request received from a client.
   *
   * @param  opContext  The operation context for the modify DN operation.
   * @param  request    The modify DN request that was received.
   */
  public void logModifyDNRequest(final OperationContext opContext,
                                 final ModifyDNRequest request)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a modify DN request that will be forwarded to another
   * server.
   *
   * @param  opContext  The operation context for the modify DN operation.
   * @param  request    The modify DN request that was received.
   * @param  target     Information about the server to which the request will
   *                    be forwarded.
   */
  public void logModifyDNForward(final OperationContext opContext,
                                 final ModifyDNRequest request,
                                 final ForwardTarget target)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a failure encountered while attempting to forward a
   * modify DN request to another server.
   *
   * @param  opContext  The operation context for the modify DN operation.
   * @param  request    The modify DN request that was received.
   * @param  target     Information about the server to which the request was
   *                    forwarded.
   * @param  failure    The exception that was received when attempting to
   *                    forward the request.
   */
  public void logModifyDNForwardFailure(final OperationContext opContext,
                                        final ModifyDNRequest request,
                                        final ForwardTarget target,
                                        final LDAPException failure)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about the result of processing a modify DN request.
   *
   * @param  opContext  The operation context for the modify DN operation.
   * @param  request    The modify DN request that was received.
   * @param  result     The result of processing the modify DN request.
   */
  public void logModifyDNResponse(final CompletedOperationContext opContext,
                                  final ModifyDNRequest request,
                                  final ModifyDNResult result)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about the result of replication assurance processing for a
   * modify DN operation.
   *
   * @param  opContext        The operation context for the modify DN operation.
   * @param  request          The modify DN request that was received.
   * @param  result           The result of processing the modify DN request.
   * @param  assuranceResult  The replication assurance processing result.
   */
  public void logModifyDNAssuranceCompleted(
                   final CompletedOperationContext opContext,
                   final ModifyDNRequest request, final ModifyDNResult result,
                   final AssuredReplicationResult assuranceResult)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a search request received from a client.
   *
   * @param  opContext  The operation context for the search operation.
   * @param  request    The search request that was received.
   */
  public void logSearchRequest(final OperationContext opContext,
                               final SearchRequest request)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a search request that will be forwarded to another
   * server.
   *
   * @param  opContext  The operation context for the search operation.
   * @param  request    The search request that was received.
   * @param  target     Information about the server to which the request will
   *                    be forwarded.
   */
  public void logSearchForward(final OperationContext opContext,
                               final SearchRequest request,
                               final ForwardTarget target)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a failure encountered while attempting to forward a
   * search request to another server.
   *
   * @param  opContext  The operation context for the search operation.
   * @param  request    The search request that was received.
   * @param  target     Information about the server to which the request was
   *                    forwarded.
   * @param  failure    The exception that was received when attempting to
   *                    forward the request.
   */
  public void logSearchForwardFailure(final OperationContext opContext,
                                      final SearchRequest request,
                                      final ForwardTarget target,
                                      final LDAPException failure)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a search result entry that was returned to the client.
   *
   * @param  opContext  The operation context for the search operation.
   * @param  request    The search request that was received.
   * @param  entry      The entry that was returned.
   * @param  controls   The set of controls included with the entry, or an empty
   *                    list if there were none.
   */
  public void logSearchResultEntry(final OperationContext opContext,
                                   final SearchRequest request,
                                   final Entry entry,
                                   final List<Control> controls)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about a search result reference that was returned to the
   * client.
   *
   * @param  opContext     The operation context for the search operation.
   * @param  request       The search request that was received.
   * @param  referralURLs  The referral URLs for the reference that was
   *                       returned.
   * @param  controls      The set of controls included with the reference, or
   *                       an empty list if there were none.
   */
  public void logSearchResultReference(final OperationContext opContext,
                                       final SearchRequest request,
                                       final List<String> referralURLs,
                                       final List<Control> controls)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about the result of processing a search request.
   *
   * @param  opContext  The operation context for the search operation.
   * @param  request    The search request that was received.
   * @param  result     The result of processing the search request.
   */
  public void logSearchResultDone(
                   final CompletedSearchOperationContext opContext,
                   final SearchRequest request, final SearchResult result)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about an unbind request received from a client.
   *
   * @param  opContext  The operation context for the unbind operation.
   * @param  request    The unbind request that was received.
   */
  public void logUnbindRequest(final OperationContext opContext,
                               final UnbindRequest request)
  {
    // No action will be taken by default.
  }



  /**
   * Logs a message about an intermediate response that was returned to the
   * client.
   *
   * @param  opContext             The operation context for the associated
   *                               operation.
   * @param  intermediateResponse  The intermediate response that was returned.
   */
  public void logIntermediateResponse(final OperationContext opContext,
                   final IntermediateResponse intermediateResponse)
  {
    // No action will be taken by default.
  }



  /**
   * Writes a message to the access logger to indicate that the Directory Proxy
   * Server will attempt to perform entry rebalancing by migrating a subtree
   * from one backend set to another.
   *
   * @param  rebalancingOperationID  The unique ID assigned to the entry
   *                                 balancing operation.
   * @param  triggerOperation        The operation that triggered the entry
   *                                 rebalancing.  It may be {@code null} if the
   *                                 rebalancing operation wasn't triggered by a
   *                                 client request.
   * @param  baseDN                  The base DN of the subtree to migrate.
   * @param  sizeLimit               The maximum number of entries to migrate.
   * @param  sourceBackendSetName    The name of the backend set containing the
   *                                 subtree to migrate.
   * @param  sourceAddress           The address of the server from which the
   *                                 source entries will be read.
   * @param  sourcePort              The port of the server from which the
   *                                 source entries will be read.
   * @param  targetBackendSetName    The name of the backend set to which the
   *                                 subtree will be migrated.
   * @param  targetAddress           The address of the server to which the
   *                                 subtree will be migrated.
   * @param  targetPort              The port of the server to which the subtree
   *                                 will be migrated.
   */
  public void logEntryRebalancingRequest(final long rebalancingOperationID,
                   final OperationContext triggerOperation, final String baseDN,
                   final int sizeLimit, final String sourceBackendSetName,
                   final String sourceAddress, final int sourcePort,
                   final String targetBackendSetName,
                   final String targetAddress, final int targetPort)
  {
    // No action performed by default.
  }



  /**
   * Writes a message to the access logger to indicate that the Directory Proxy
   * Server will attempt to perform entry rebalancing by migrating a subtree
   * from one backend set to another.
   *
   * @param  rebalancingOperationID  The unique ID assigned to the entry
   *                                 balancing operation.
   * @param  triggerOperation        The operation that triggered the entry
   *                                 rebalancing.  It may be {@code null} if the
   *                                 rebalancing operation wasn't triggered by a
   *                                 client request.
   * @param  baseDN                  The base DN of the subtree to migrate.
   * @param  sizeLimit               The maximum number of entries to migrate.
   * @param  sourceBackendSetName    The name of the backend set containing the
   *                                 subtree to migrate.
   * @param  sourceAddress           The address of the server from which the
   *                                 source entries will be read.
   * @param  sourcePort              The port of the server from which the
   *                                 source entries will be read.
   * @param  targetBackendSetName    The name of the backend set to which the
   *                                 subtree will be migrated.
   * @param  targetAddress           The address of the server to which the
   *                                 subtree will be migrated.
   * @param  targetPort              The port of the server to which the subtree
   *                                 will be migrated.
   * @param  moveSubtreeResult       An object with information about the result
   *                                 of the subtree move processing.
   */
  public void logEntryRebalancingResult(final long rebalancingOperationID,
                   final OperationContext triggerOperation, final String baseDN,
                   final int sizeLimit, final String sourceBackendSetName,
                   final String sourceAddress, final int sourcePort,
                   final String targetBackendSetName,
                   final String targetAddress, final int targetPort,
                   final MoveSubtreeResult moveSubtreeResult)
  {
    // No action performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
