/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2016-2019 Ping Identity Corporation
 */


package com.unboundid.directory.sdk.common.api;

import com.unboundid.directory.sdk.broker.internal.BrokerExtension;
import com.unboundid.directory.sdk.common.config.AccessTokenValidatorConfig;
import com.unboundid.directory.sdk.common.types.TokenValidationResult;
import com.unboundid.directory.sdk.common.internal.Configurable;
import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.types.ServerContext;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.directory.sdk.metrics.internal.MetricsEngineExtension;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;

import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * This class defines an API that may be implemented by ${BASE_NAME}
 * extensions that validate externally generated access tokens.  Implementing
 * extensions that support this API enables the ${BASE_NAME}
 * to accept access tokens generated from external Identity Providers.
 *
 * <H2>Configuring Access Token Validators</H2>
 * In order to configure an Access Token Validator created using this API, use
 * a command like:
 * <PRE>
 *      dsconfig create-token-validator \
 *           ---validator-name "<I>{name}</I>" \
 *           --type third-party \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{name}</I>" is the name to use for the Access Token Validator
 * instance, "<I>{class-name}</I>" is the fully-qualified name of the Java class
 * that extends
 * {@code com.unboundid.directory.sdk.common.api.AccessTokenValidator},
 * and "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the Access Token Validator. If multiple arguments should be
 * provided to the extension, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 */
@Extensible()
@BrokerExtension
@DirectoryServerExtension
@SynchronizationServerExtension(appliesToLocalContent=true,
  appliesToSynchronizedContent=false)
@MetricsEngineExtension
@DirectoryProxyServerExtension(
    appliesToLocalContent = true,
    appliesToRemoteContent = true
)
@ThreadSafety(level= ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class AccessTokenValidator
    implements UnboundIDExtension, Configurable, ExampleUsageProvider {

  /**
   * Creates a new instance of this Access Token Validator.  All
   * implementations must include a default constructor, but any
   * initialization should generally be done in the
   * {@link #initializeTokenValidator} method.
   */
  public AccessTokenValidator()
  {
    // No implementation is required.
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  @Override
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  @Override
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void defineConfigArguments(final ArgumentParser parser)
      throws ArgumentException
  {
    // No arguments will be allowed by default.
  }


  /**
   * Initializes this access token validator.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this token validator.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this token validator.
   *
   * @throws Exception      If a problem occurs while initializing this
   *                        token validator.
   */
  public void initializeTokenValidator(
      final ServerContext serverContext,
      final AccessTokenValidatorConfig config,
      final ArgumentParser parser)
      throws Exception
  {
    // No initialization will be performed by default.
  }


  /**
   * Performs any cleanup which may be necessary when this token validator
   * is to be taken out of service.
   */
  public void finalizeTokenValidator()
  {
    // No implementation is performed by default.
  }


  /**
   * Validate the provided access token.
   * @param encodedAccessToken access token string as it is received from the
   *                           requesting client.
   * @return The ${BASE_NAME} may be configured to accept access tokens
   * from multiple sources so it is important that each validator differentiate
   * between a token format that it does not recognize and a token that it can
   * process but is not valid.
   *
   * If the token can be processed, the validator must return a
   * TokenValidationResult object containing token properties.  Most
   * importantly the {@code active} field of the TokenValidationResult must be
   * set by the validator.
   *
   * The decision as to whether an access token is accepted or not is made by
   * the servlet hosting the token validator.
   *
   * If the token cannot be introspected by the Access Token Validator it must
   * return null to allow other validators to have a chance to process the
   * token.
   *
   * @throws Exception if an error occurs during the processing of a token
   * that can be introspected by the validator.  Exceptions should only be
   * thrown for unexpected internal errors.   Sensitive information should not
   * be included in the exception message as the message may be returned to
   * the client application that has passed the token.
   */
  public abstract TokenValidationResult validate(String encodedAccessToken)
      throws Exception;

}
