/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.unboundid.directory.sdk.broker.internal.BrokerExtension;
import com.unboundid.directory.sdk.common.config.ErrorLoggerConfig;
import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.types.LogCategory;
import com.unboundid.directory.sdk.common.types.LogSeverity;
import com.unboundid.directory.sdk.common.types.ServerContext;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.metrics.internal.MetricsEngineExtension;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which
 * record information about warnings, errors, and events which occur in the
 * server.
 * <BR><BR>
 * Each error logger may be configured to indicate whether to include or
 * exclude log messages based on their category and/or severity.  This is
 * handled automatically by the server, so individual error logger
 * implementations no not need to attempt to perform that filtering on their
 * own.  However, they may perform additional processing if desired to further
 * narrow the set of messages that should be logged.
 * <BR>
 * <H2>Configuring Error Loggers</H2>
 * In order to configure an error logger created using this API, use a command
 * like:
 * <PRE>
 *      dsconfig create-log-publisher \
 *           --publisher-name "<I>{logger-name}</I>" \
 *           --type third-party-error \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{logger-name}</I>" is the name to use for the error logger
 * instance, "<I>{class-name}</I>" is the fully-qualified name of the Java class
 * that extends {@code com.unboundid.directory.sdk.common.api.ErrorLogger},
 * and "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the logger.  If multiple arguments should be provided to the
 * logger, then the "<CODE>--set extension-argument:<I>{name=value}</I></CODE>"
 * option should be provided multiple times.
 *
 * @see  FileBasedErrorLogger
 * @see  com.unboundid.directory.sdk.common.scripting.ScriptedErrorLogger
 * @see
 *    com.unboundid.directory.sdk.common.scripting.ScriptedFileBasedErrorLogger
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=true)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=true)
@MetricsEngineExtension()
@BrokerExtension()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class ErrorLogger
       implements UnboundIDExtension, Reconfigurable<ErrorLoggerConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this error logger.  All error logger
   * implementations must include a default constructor, but any initialization
   * should generally be done in the {@code initializeErrorLogger} method.
   */
  public ErrorLogger()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this error logger.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this error logger.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this error logger.
   *
   * @throws  LDAPException  If a problem occurs while initializing this error
   *                         logger.
   */
  public void initializeErrorLogger(final ServerContext serverContext,
                                    final ErrorLoggerConfig config,
                                    final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(final ErrorLoggerConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final ErrorLoggerConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this error logger " +
                "is disabled and re-enabled or until the server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this error logger is to
   * be taken out of service.
   */
  public void finalizeErrorLogger()
  {
    // No implementation is required.
  }



  /**
   * Records information about the provided message, if appropriate.
   *
   * @param  category   The category for the message to be logged.
   * @param  severity   The severity for the message to be logged.
   * @param  messageID  The unique identifier with which the message text is
   *                    associated.
   * @param  message    The message to be logged.
   */
  public abstract void logError(final LogCategory category,
                                final LogSeverity severity,
                                final long messageID,
                                final String message);



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
