/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.net.ssl.KeyManager;

import com.unboundid.directory.sdk.broker.internal.BrokerExtension;
import com.unboundid.directory.sdk.common.config.KeyManagerProviderConfig;
import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.types.ServerContext;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.metrics.internal.MetricsEngineExtension;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which
 * provide access to key managers.  Key managers allow the server to access
 * certificates in a form that can be presented to another system during SSL or
 * StartTLS negotiation.  If the server is configured to accept secure
 * communication from clients, then a key manager provider will be used to
 * access the certificate that the server presents to the client.  If the server
 * needs to establish a secure connection to another system (e.g., the Directory
 * Proxy Server connecting to a backend Directory Server instance), then the
 * key manager provider may also be used to obtain a client certificate that may
 * be used for authentication.
 * <BR>
 * <H2>Configuring Key Manager Providers</H2>
 * In order to configure a key manager provider created using this API, use a
 * command like:
 * <PRE>
 *      dsconfig create-key-manager-provider \
 *           --provider-name "<I>{provider-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{provider-name}</I>" is the name to use for the key manager
 * provider instance, "<I>{class-name}</I>" is the fully-qualified name of the
 * Java class that extends
 * {@code com.unboundid.directory.sdk.common.api.KeyManagerProvider}, and
 * "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the key manager provider.  If multiple arguments should be
 * provided to the key manager provider, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=false)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@MetricsEngineExtension()
@BrokerExtension()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class KeyManagerProvider
       implements UnboundIDExtension, Reconfigurable<KeyManagerProviderConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this key manager provider.  All key manager
   * provider implementations must include a default constructor, but any
   * initialization should generally be done in the
   * {@code initializeKeyManagerProvider} method.
   */
  public KeyManagerProvider()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this key manager provider.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this key manager
   *                        provider.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this key manager provider.
   *
   * @throws  LDAPException  If a problem occurs while initializing this
   *                         key manager provider.
   */
  public void initializeKeyManagerProvider(final ServerContext serverContext,
                   final KeyManagerProviderConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final KeyManagerProviderConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final KeyManagerProviderConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this key manager " +
                "provider is disabled and re-enabled or until the server is " +
                "restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this key manager provider
   * is to be taken out of service.
   */
  public void finalizeKeyManagerProvider()
  {
    // No implementation is required.
  }



  /**
   * Retrieves a set of key managers that may be used for operations within
   * the server which may require access to certificates.
   *
   * @return  The set of key managers that may be used for operations within the
   *          server which may require access to certificates.
   *
   * @throws  LDAPException  If a problem occurs while attempting to retrieve
   *                         the key managers.
   */
  public abstract KeyManager[] getKeyManagers()
         throws LDAPException;



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
