/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2011-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.api;

import com.unboundid.directory.sdk.broker.internal.BrokerExtension;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.types.InstallExtensionDetails;
import com.unboundid.directory.sdk.common.types.PostManageExtensionPluginResult;
import com.unboundid.directory.sdk.common.types.PreManageExtensionPluginResult;
import com.unboundid.directory.sdk.common.types.UpdateExtensionDetails;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.metrics.internal.MetricsEngineExtension;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;

/**
 * This class defines an API that may be implemented by extensions which will
 * be managed by the manage-extension tool. This API allows extensions to
 * perform custom operations during the install/update process.
 * <BR><BR>
 * This plugin will be invoked during the following phases:
 * <UL>
 * <LI>before install -- Invoked when an extension is first installed before
 *     any files are copied to the server installation.</LI>
 * <LI>after install -- Invoked when an extension is first installed after
 *     all files are copied to the server installation.</LI>
 * <LI>before update -- Invoked when an extension is updated before any files
 *     are updated.</LI>
 * <LI>after update -- Invoked when an extension is updated after all files
 *     are updated.</LI>
 * </UL>
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=true)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@MetricsEngineExtension()
@BrokerExtension()
@ThreadSafety(level= ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class ManageExtensionPlugin implements UnboundIDExtension
{
  /**
   * Invoked when an extension is first installed before shutting down
   * the server and any files are copied to the server installation.
   *
   * @param details The state of the install before any files are copied.
   * @return Whether to continue the installation process.
   */
  public PreManageExtensionPluginResult preInstall(
      final InstallExtensionDetails details)
  {
    // No default implementation
    return PreManageExtensionPluginResult.SUCCESS;
  }

  /**
   * Invoked when an extension is first installed after all files are copied
   * to the server installation but before restarting the server.
   *
   * @param details The state of the install after all files are copied.
   * @return Whether the server should be restarted after the install if
   *                 previous started.
   */
  public PostManageExtensionPluginResult postInstall(
      final InstallExtensionDetails details)
  {
    // No default implementation
    return PostManageExtensionPluginResult.SUCCESS;
  }

  /**
   * Invoked when an extension is updated before shutting down the server and
   * any files are updated to the server installation.
   *
   * @param details The state of the update before any files are updated.
   * @return Whether to continue the update process.
   */
  public PreManageExtensionPluginResult preUpdate(
      final UpdateExtensionDetails details)
  {
    // No default implementation
    return PreManageExtensionPluginResult.SUCCESS;
  }

  /**
   * Invoked when an extension is updated after all files are update to the
   * server installation but before restarting the server.
   *
   * @param details The state of the update after all files are updated.
   * @return Whether the server should be restarted after the update if
   *                 previous started.
   */
  public PostManageExtensionPluginResult postUpdate(
      final UpdateExtensionDetails details)
  {
    // No default implementation
    return PostManageExtensionPluginResult.SUCCESS;
  }
}
