/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.unboundid.directory.sdk.broker.internal.BrokerExtension;
import com.unboundid.directory.sdk.common.config.MonitorProviderConfig;
import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.types.ServerContext;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.metrics.internal.MetricsEngineExtension;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.Attribute;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which
 * expose monitoring information about server components.  Each monitor provider
 * instance will be used by the server to construct a single monitor entry.
 * The monitor information will be obtained on demand whenever the server
 * receives a request targeting a monitor entry, but monitor providers may also
 * indicate that they should be periodically invoked by a background thread so
 * that they can collect additional information on a regular basis even when no
 * clients have requested monitor information.
 * <BR>
 * <H2>Configuring Monitor Providers</H2>
 * In order to configure a monitor provider created using this API, use a
 * command like:
 * <PRE>
 *      dsconfig create-monitor-provider \
 *           --provider-name "<I>{provider-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{provider-name}</I>" is the name to use for the monitor provider
 * instance, "<I>{class-name}</I>" is the fully-qualified name of the Java class
 * that extends {@code com.unboundid.directory.sdk.common.api.MonitorProvider},
 * and "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the monitor provider.  If multiple arguments should be provided to
 * the monitor provider, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=false)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@MetricsEngineExtension()
@BrokerExtension()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class MonitorProvider
       implements UnboundIDExtension, Reconfigurable<MonitorProviderConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this monitor provider.  All monitor provider
   * implementations must include a default constructor, but any initialization
   * should generally be done in the {@code initializeMonitorProvider} method.
   */
  public MonitorProvider()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this monitor provider.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this monitor provider.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this monitor provider.
   *
   * @throws  LDAPException  If a problem occurs while initializing this monitor
   *                         provider
   */
  public void initializeMonitorProvider(final ServerContext serverContext,
                                        final MonitorProviderConfig config,
                                        final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(final MonitorProviderConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final MonitorProviderConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this monitor " +
                "provider is disabled and re-enabled or until the server is " +
                "restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this monitor provider is
   * to be taken out of service.
   */
  public void finalizeMonitorProvider()
  {
    // No implementation is required.
  }



  /**
   * Retrieves the name that identifies this monitor provider instance.  It
   * will be used as the value of the naming attribute for monitor entries.
   * Each monitor provider instance must have a unique name.
   *
   * @return  The name that identifies this monitor provider instance.
   */
  public abstract String getMonitorInstanceName();



  /**
   * Retrieves the name of the object class that will be used for monitor
   * entries created from this monitor provider. This does not need to be
   * defined in the server schema. It may be {@code null} if a default object
   * class should be used.
   *
   * @return  The name of the object class that will be used for monitor entries
   *          created from this monitor provider.
   */
  public String getMonitorObjectClass()
  {
    return null;
  }



  /**
   * Retrieves the update interval in milliseconds that should be used for this
   * monitor provider.  A value that is greater than zero will cause the
   * {@link #updateMonitorData} method to be repeatedly invoked at that
   * interval.  A value less than or equal to zero indicates that the monitor
   * provider should not be periodically updated.
   * <p>
   * If the initial returned value is greater than zero, then this method will
   * be invoked between each update to see if the update interval has changed.
   * This way you can change the interval dynamically.
   *
   * @return  The update interval in milliseconds that should be used for this
   *          monitor provider.
   */
  public long getUpdateIntervalMillis()
  {
    // This monitor provider will not be periodically updated by default.
    return 0L;
  }



  /**
   * Updates the information collected by this monitor provider.  This method
   * will be periodically invoked if the {@link #getUpdateIntervalMillis} method
   * returns a positive value.
   */
  public void updateMonitorData()
  {
    // No implementation provided by default.
  }



  /**
   * Retrieves a list of attributes containing the data to include in the
   * monitor entry generated from this monitor provider.
   *
   * @return  A list of attributes containing the data to include in the monitor
   *          entry generated from this monitor provider.
   */
  public abstract List<Attribute> getMonitorAttributes();



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
