/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.net.ssl.TrustManager;

import com.unboundid.directory.sdk.broker.internal.BrokerExtension;
import com.unboundid.directory.sdk.common.config.TrustManagerProviderConfig;
import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.types.ServerContext;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.metrics.internal.MetricsEngineExtension;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which
 * provide access to trust managers which are used to determine whether to trust
 * a certificate that has been presented to the server.  Trust managers are
 * generally used when performing cryptographic operations, including SSL and
 * StartTLS communication, in which a certificate is presented to the server.
 * In such cases, the secure communication will only be allowed if the trust
 * managers determine that the presented certificate chain is trustworthy.
 * <BR>
 * <H2>Configuring Trust Manager Providers</H2>
 * In order to configure a trust manager provider created using this API, use a
 * command like:
 * <PRE>
 *      dsconfig create-trust-manager-provider \
 *           --provider-name "<I>{provider-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{provider-name}</I>" is the name to use for the trust manager
 * provider instance, "<I>{class-name}</I>" is the fully-qualified name of the
 * Java class that extends
 * {@code com.unboundid.directory.sdk.common.api.TrustManagerProvider}, and
 * "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the trust manager provider.  If multiple arguments should be
 * provided to the trust manager provider, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=false)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@MetricsEngineExtension()
@BrokerExtension()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class TrustManagerProvider
       implements UnboundIDExtension,
                  Reconfigurable<TrustManagerProviderConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this trust manager provider.  All trust manager
   * provider implementations must include a default constructor, but any
   * initialization should generally be done in the
   * {@code initializeTrustManagerProvider} method.
   */
  public TrustManagerProvider()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this trust manager provider.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this trust manager
   *                        provider.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this trust manager provider.
   *
   * @throws  LDAPException  If a problem occurs while initializing this
   *                         trust manager provider.
   */
  public void initializeTrustManagerProvider(final ServerContext serverContext,
                   final TrustManagerProviderConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final TrustManagerProviderConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final TrustManagerProviderConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this trust manager " +
                "provider is disabled and re-enabled or until the server is " +
                "restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this trust manager
   * provider is to be taken out of service.
   */
  public void finalizeTrustManagerProvider()
  {
    // No implementation is required.
  }



  /**
   * Retrieves a set of trust managers that may be used for operations within
   * the server which may need to determine whether to trust a presented
   * certificate chain.
   *
   * @return  The set of trust managers that may be used for operations within
   *          the server which may need to determine whether to trust a
   *          presented certificate chain.
   *
   * @throws  LDAPException  If a problem occurs while attempting to retrieve
   *                         the trust managers.
   */
  public abstract TrustManager[] getTrustManagers()
         throws LDAPException;



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
