/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.operation;



import com.unboundid.ldap.sdk.ReadOnlyDeleteRequest;
import com.unboundid.ldap.sdk.unboundidds.controls.HardDeleteRequestControl;
import com.unboundid.ldap.sdk.unboundidds.controls.SoftDeleteRequestControl;
import com.unboundid.ldif.LDIFDeleteChangeRecord;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods which may be used to interact with a
 * delete request.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_NOT_THREADSAFE)
public interface DeleteRequest
       extends ChangeRequest
{
  /**
   * Retrieves the DN of the entry to be deleted.
   *
   * @return  The DN of the entry to be deleted.
   */
  String getDN();



  /**
   * Retrieves an LDAP SDK representation of this delete request.
   *
   * @return  An LDAP SDK representation of this delete request.
   */
  ReadOnlyDeleteRequest toLDAPSDKRequest();



  /**
   * Retrieves an LDIF change record that corresponds to this delete request.
   *
   * @return  An LDIF change record that corresponds to this delete request.
   */
  LDIFDeleteChangeRecord toLDIFChangeRecord();



  /**
   * Indicates whether this request will result in the target entry being
   * soft-deleted. This will be true either if the
   * {@link SoftDeleteRequestControl} is included or if the request matches
   * the auto-soft-delete connection criteria or request criteria specified in
   * the Soft Delete Policy for the server.
   *
   * @return true if this request will result in the target entry being
   *         soft-deleted, false otherwise.
   */
  boolean isSoftDelete();



  /**
   * Retrieves the soft-delete request control included in the delete request if
   * it is present. Note that soft delete operations can occur without the
   * {@link SoftDeleteRequestControl} if the request matches the
   * auto-soft-delete connection criteria; in that case this method will return
   * {@code null}.
   *
   * @return the soft-delete request control if it is present in the request,
   *         otherwise {@code null}.
   */
  SoftDeleteRequestControl getSoftDeleteRequestControl();



  /**
   * Indicates whether this request will result in the target entry being
   * hard-deleted. This will be true either if the
   * {@link HardDeleteRequestControl} is included or if the request didn't match
   * any condition necessary to make it a soft-delete.
   *
   * @return true if this request will result in the target entry being
   *         hard-deleted, false otherwise.
   */
  boolean isHardDelete();



  /**
   * Retrieves the hard-delete request control included in the delete request if
   * it is present. Note that hard delete operations can occur without the
   * {@link HardDeleteRequestControl}. If the request is not a soft-delete, it
   * is effectively a hard delete; the presence of the
   * {@link HardDeleteRequestControl} just makes it explicit.
   *
   * @return the hard-delete request control if it is present in the request,
   *         otherwise {@code null}.
   */
  HardDeleteRequestControl getHardDeleteRequestControl();
}
