/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.schema;



import com.unboundid.util.ByteString;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface provides an API for interacting with attribute syntaxes, which
 * can be used to constrain the kinds of data that may be stored in associated
 * attributes.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface AttributeSyntax
{
  /**
   * Retrieves the numeric OID for this attribute syntax.
   *
   * @return  The numeric OID for this attribute syntax.
   */
  String getOID();



  /**
   * Retrieves the name for this attribute syntax, if any.
   *
   * @return  The name for this attribute syntax, or {@code null} if it does not
   *          have a name.
   */
  String getName();



  /**
   * Retrieves the name for this attribute syntax, or the numeric OID if it does
   * not have a name.
   *
   * @return  The name for this attribute syntax, or the numeric OID if it does
   *          not have a name.
   */
  String getNameOrOID();



  /**
   * Indicates whether the provided string matches the name or numeric OID for
   * this attribute syntax.
   *
   * @param  name  The name for which to make the determination.
   *
   * @return  {@code true} if the provided string matches the name or numeric
   *          OID for this attribute syntax, or {@code false} if not.
   */
  boolean hasNameOrOID(final String name);



  /**
   * Retrieves the description for this attribute syntax, if any.
   *
   * @return  The description for this attribute syntax, or {@code null} if it
   *          does not have a description.
   */
  String getDescription();



  /**
   * Retrieves the default equality matching rule that will be used for
   * attribute types with this syntax which do not explicitly specify their own
   * equality matching rule.
   *
   * @return  The default equality matching rule that will be used for attribute
   *          types with this syntax, or {@code null} if there is no default
   *          equality matching rule.
   */
  EqualityMatchingRule getDefaultEqualityMatchingRule();



  /**
   * Retrieves the default ordering matching rule that will be used for
   * attribute types with this syntax which do not explicitly specify their own
   * ordering matching rule.
   *
   * @return  The default ordering matching rule that will be used for attribute
   *          types with this syntax, or {@code null} if there is no default
   *          ordering matching rule.
   */
  OrderingMatchingRule getDefaultOrderingMatchingRule();



  /**
   * Retrieves the default substring matching rule that will be used for
   * attribute types with this syntax which do not explicitly specify their own
   * substring matching rule.
   *
   * @return  The default substring matching rule that will be used for
   *          attribute types with this syntax, or {@code null} if there is no
   *          default substring matching rule.
   */
  SubstringMatchingRule getDefaultSubstringMatchingRule();



  /**
   * Retrieves the default approximate matching rule that will be used for
   * attribute types with this syntax which do not explicitly specify their own
   * approximate matching rule.
   *
   * @return  The default approximate matching rule that will be used for
   *          attribute types with this syntax, or {@code null} if there is no
   *          default approximate matching rule.
   */
  ApproximateMatchingRule getDefaultApproximateMatchingRule();



  /**
   * Indicates whether the provided value is acceptable for use with this
   * attribute syntax.  NOTE:  It is recommended that the
   * {@link #valueIsAcceptable(AttributeType, ByteString, StringBuilder)}
   * version of this method be used when possible.
   *
   * @param  value          The value for which to make the determination.
   * @param  invalidReason  A buffer to which a message may be appended with
   *                        information about why the given value is not
   *                        acceptable.
   *
   * @return  {@code true} if the provided value is acceptable for use with this
   *          attribute syntax, or {@code false} if not.
   */
  boolean valueIsAcceptable(final ByteString value,
                            final StringBuilder invalidReason);



  /**
   * Indicates whether the provided value is acceptable for use with this
   * attribute syntax.
   *
   * @param  attributeType  The attribute type with which the value is
   *                        associated.
   * @param  value          The value for which to make the determination.
   * @param  invalidReason  A buffer to which a message may be appended with
   *                        information about why the given value is not
   *                        acceptable.
   *
   * @return  {@code true} if the provided value is acceptable for use with this
   *          attribute syntax, or {@code false} if not.
   */
  boolean valueIsAcceptable(final AttributeType attributeType,
                            final ByteString value,
                            final StringBuilder invalidReason);



  /**
   * Retrieves the hash code for this attribute syntax.
   *
   * @return  The hash code for this attribute syntax.
   */
  int hashCode();



  /**
   * Indicates whether the provided object is equal to this attribute syntax.
   *
   * @param  o  The object for which to make the determination.
   *
   * @return  {@code true} if the provided object is equal to this attribute
   *          syntax, or {@code false} if not.
   */
  boolean equals(final Object o);



  /**
   * Retrieves a string representation of this attribute syntax.
   *
   * @return  A string representation of this attribute syntax.
   */
  String toString();
}
