/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.schema;



import java.util.List;
import java.util.Map;

import com.unboundid.ldap.sdk.schema.AttributeUsage;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be used to obtain
 * information about an attribute type defined in the server schema.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface AttributeType
{
  /**
   * Retrieves the numeric OID for this attribute type.
   *
   * @return  The numeric OID for this attribute type.
   */
  String getOID();



  /**
   * Retrieves the list of names for this attribute type, if any.
   *
   * @return  The list of names for this attribute type, or an empty list if
   *          there are no user-defined names.
   */
  List<String> getNames();



  /**
   * Retrieves the primary name for this attribute type, or the numeric OID if
   * no names are defined.
   *
   * @return The primary name or OID for this attribute type.
   */
  String getNameOrOID();



  /**
   * Indicates whether the provided string is equal to the OID or any of the
   * defined names for this attribute type.
   *
   * @param  name  The name for which to make the determination.
   *
   * @return  {@code true} if the provided string matches the OID or one of the
   *          names for this attribute type, or {@code false} if not.
   */
  boolean hasNameOrOID(final String name);



  /**
   * Retrieves the description for this attribute type, if any.
   *
   * @return  The description for this attribute type, or {@code null} if it
   *          does not have a description.
   */
  String getDescription();



  /**
   * Retrieves the superior type for this attribute type, if any.
   *
   * @return  The superior type for this attribute type, or {@code null} if this
   *          attribute type does not have a superior type.
   */
  AttributeType getSuperiorType();



  /**
   * Retrieves the attribute syntax for this attribute type.
   *
   * @return  The attribute syntax for this attribute type.
   */
  AttributeSyntax getSyntax();



  /**
   * Retrieves the equality matching rule for this attribute type, if any.
   *
   * @return  The equality matching rule for this attribute type, or
   *          {@code null} if there is no equality matching rule for this
   *          attribute type.
   */
  EqualityMatchingRule getEqualityMatchingRule();



  /**
   * Retrieves the ordering matching rule for this attribute type, if any.
   *
   * @return  The ordering matching rule for this attribute type, or
   *          {@code null} if there is no ordering matching rule for this
   *          attribute type.
   */
  OrderingMatchingRule getOrderingMatchingRule();



  /**
   * Retrieves the substring matching rule for this attribute type, if any.
   *
   * @return  The substring matching rule for this attribute type, or
   *          {@code null} if there is no substring matching rule for this
   *          attribute type.
   */
  SubstringMatchingRule getSubstringMatchingRule();



  /**
   * Retrieves the approximate matching rule for this attribute type, if any.
   *
   * @return  The approximate matching rule for this attribute type, or
   *          {@code null} if there is no approximate matching rule for this
   *          attribute type.
   */
  ApproximateMatchingRule getApproximateMatchingRule();



  /**
   * Retrieves the usage for this attribute type.
   *
   * @return  The usage for this attribute type.
   */
  AttributeUsage getUsage();



  /**
   * Indicates whether this attribute type has an operational usage.
   *
   * @return  {@code true} if this attribute type has an operational usage, or
   *          {@code false} if not.
   */
  boolean isOperational();



  /**
   * Indicates whether this attribute type is declared single-valued in the
   * server schema.
   *
   * @return  {@code true} if this attribute type is declared single-valued in
   *          the server schema, or {@code false} if not.
   */
  boolean isSingleValued();



  /**
   * Indicates whether this attribute type is declared collective in the server
   * schema.
   *
   * @return  {@code true} if this attribute type is declared collective in the
   *          server schema, or {@code false} if not.
   */
  boolean isCollective();



  /**
   * Indicates whether this attribute type is declared obsolete in the server
   * schema.
   *
   * @return  {@code true} if this attribute type is declared obsolete in the
   *          server schema, or {@code false} if not.
   */
  boolean isObsolete();



  /**
   * Indicates whether this attribute type is declared NO-USER-MODIFICATION in
   * the server schema.
   *
   * @return  {@code true} if this attribute type is declared
   *          NO-USER-MODIFICATION in the server schema, or {@code false} if
   *          not.
   */
  boolean isNoUserModification();



  /**
   * Retrieves a map of all defined extensions for this attribute type.
   *
   * @return  A map of all defined extensions for this attribute type.
   */
  Map<String,List<String>> getExtensions();



  /**
   * Retrieves the name of the schema file in which this attribute type is
   * defined.
   *
   * @return  The name of the schema file in which this attribute type is
   *          defined.
   */
  String getSchemaFileName();



  /**
   * Indicates whether the provided object is equal to this attribute type.
   *
   * @param  o  The object for which to make the determination.
   *
   * @return  {@code true} if the provided object is equal to this attribute
   *          type, or {@code false} if not.
   */
  boolean equals(final Object o);



  /**
   * Retrieves a hash code for this attribute type.
   *
   * @return  A hash code for this attribute type.
   */
  int hashCode();



  /**
   * Retrieves a string representation of this attribute type definition.
   *
   * @return  A string representation of this attribute type definition.
   */
  String toString();
}
