/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.schema;



import java.util.List;
import java.util.Map;
import java.util.Set;

import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be used to obtain
 * information about a DIT structure rule defined in the server schema.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface DITStructureRule
{
  /**
   * Retrieves the rule ID for this DIT structure rule.
   *
   * @return  The rule ID for this DIT structure rule.
   */
  int getRuleID();



  /**
   * Retrieves the list of names for this DIT structure rule, if any.
   *
   * @return  The list of names for this DIT structure rule, or an empty list if
   *          there are no user-defined names.
   */
  List<String> getNames();



  /**
   * Retrieves the primary name for this DIT structure rule, or the rule ID if
   * no names are defined.
   *
   * @return The primary name or rule ID for this DIT structure rule.
   */
  String getNameOrRuleID();



  /**
   * Indicates whether the provided string is equal to any of the defined names
   * or the rule ID for this DIT structure rule.
   *
   * @param  name  The name for which to make the determination.
   *
   * @return  {@code true} if the provided string matches one of the names or
   *          the rule ID for this DIT structure rule, or {@code false} if not.
   */
  boolean hasNameOrRuleID(final String name);



  /**
   * Retrieves the description for this DIT content rule, if any.
   *
   * @return  The description for this DIT content rule, or {@code null} if it
   *          does not have a description.
   */
  String getDescription();



  /**
   * Retrieves the name form for this DIT structure rule.
   *
   * @return  The name form for this DIT structure rule.
   */
  NameForm getNameForm();



  /**
   * Retrieves the structural object class for this DIT content rule.
   *
   * @return  The structural object class for this DIT content rule.
   */
  ObjectClass getStructuralClass();



  /**
   * Retrieves the set of superior DIT structure rules for this rule, if any.
   *
   * @return  The set of superior DIT structure rules, or an empty set if there
   *          are no superior rules.
   */
  Set<DITStructureRule> getSuperiorRules();



  /**
   * Indicates whether this DIT structure rule is declared obsolete in the
   * server schema.
   *
   * @return  {@code true} if this DIT structure rule is declared obsolete in
   *          the server schema, or {@code false} if not.
   */
  boolean isObsolete();



  /**
   * Retrieves a map of all defined extensions for this DIT structure rule.
   *
   * @return  A map of all defined extensions for this DIT structure rule.
   */
  Map<String,List<String>> getExtensions();



  /**
   * Retrieves the name of the schema file in which this DIT structure rule is
   * defined.
   *
   * @return  The name of the schema file in which this DIT structure rule is
   *          defined.
   */
  String getSchemaFileName();



  /**
   * Indicates whether the provided object is equal to this DIT structure rule.
   *
   * @param  o  The object for which to make the determination.
   *
   * @return  {@code true} if the provided object is equal to this DIT structure
   *          rule, or {@code false} if not.
   */
  boolean equals(final Object o);



  /**
   * Retrieves a hash code for this DIT structure rule.
   *
   * @return  A hash code for this DIT structure rule.
   */
  int hashCode();



  /**
   * Retrieves a string representation of this DIT structure rule definition.
   *
   * @return  A string representation of this DIT structure rule definition.
   */
  String toString();
}
