/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.schema;



import com.unboundid.directory.sdk.common.types.ConditionResult;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.util.ByteString;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface provides an API for interacting with matching rules, which
 * can be used to perform matching-related operations against data.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface MatchingRule
{
  /**
   * Retrieves the numeric OID for this matching rule.
   *
   * @return  The numeric OID for this matching rule.
   */
  String getOID();



  /**
   * Retrieves the name for this matching rule, if any.
   *
   * @return  The name for this matching rule, or {@code null} if it does not
   *          have a name.
   */
  String getName();



  /**
   * Retrieves the name for this matching rule, or the numeric OID if it does
   * not have a name.
   *
   * @return  The name for this matching rule, or the numeric OID if it does not
   *          have a name.
   */
  String getNameOrOID();



  /**
   * Indicates whether the provided string matches the name or numeric OID for
   * this matching rule.
   *
   * @param  name  The name for which to make the determination.
   *
   * @return  {@code true} if the provided string matches the name or numeric
   *          OID for this matching rule, or {@code false} if not.
   */
  boolean hasNameOrOID(final String name);



  /**
   * Retrieves the description for this matching rule, if any.
   *
   * @return  The description for this matching rule, or {@code null} if it does
   *          not have a description.
   */
  String getDescription();



  /**
   * Retrieves the OID of the attribute syntax with which this matching rule is
   * most closely associated.
   *
   * @return  The OID of the attribute syntax with which this matching rule is
   *          most closely associated.
   */
  String getSyntaxOID();



  /**
   * Indicates whether this matching rule is declared obsolete in the server
   * schema.
   *
   * @return  {@code true} if this matching rule is declared obsolete, or
   *          {@code false} if not.
   */
  boolean isObsolete();



  /**
   * Retrieves the normalized form of the provided value.
   *
   * @param  value  The value to be normalized.
   *
   * @return  The normalized form of the provided value.
   *
   * @throws  LDAPException  If an error occurs while attempting to normalize
   *                         the provided value (e.g., it does not conform to
   *                         the appropriate syntax).
   */
  ByteString normalizeValue(final ByteString value)
       throws LDAPException;



  /**
   * Retrieves the normalized form of the provided value.
   *
   * @param  type   The associated attribute type.
   * @param  value  The value to be normalized.
   *
   * @return  The normalized form of the provided value.
   *
   * @throws  LDAPException  If an error occurs while attempting to normalize
   *                         the provided value (e.g., it does not conform to
   *                         the appropriate syntax).
   */
  ByteString normalizeValue(final AttributeType type, final ByteString value)
       throws LDAPException;



  /**
   * Indicates whether the provided attribute value may be considered logically
   * equivalent to the provided assertion value according to the constraints of
   * this matching rule..
   *
   * @param  normAttributeValue  The normalized form of the attribute value to
   *                             be compared.
   * @param  normAssertionValue  The normalized form of the assertion value to
   *                             be compared.
   *
   * @return  {@code TRUE} if the values are considered equal,
   *          {@code FALSE} if the values are considered different, or
   *          {@code UNDEFINED} if the result is not defined for this matching
   *          rule.
   */
  ConditionResult valuesMatch(final ByteString normAttributeValue,
                              final ByteString normAssertionValue);



  /**
   * Retrieves the hash code for this matching rule.
   *
   * @return  The hash code for this matching rule.
   */
  int hashCode();



  /**
   * Indicates whether the provided object is equal to this matching rule.
   *
   * @param  o  The object for which to make the determination.
   *
   * @return  {@code true} if the provided object is equal to this matching
   *          rule, or {@code false} if not.
   */
  boolean equals(final Object o);



  /**
   * Retrieves a string representation of this matching rule.
   *
   * @return  A string representation of this matching rule.
   */
  String toString();
}
