/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.schema;



import java.util.List;
import java.util.Map;
import java.util.Set;

import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be used to obtain
 * information about a matching rule use defined in the server schema.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface MatchingRuleUse
{
  /**
   * Retrieves the list of names for this matching rule use, if any.
   *
   * @return  The list of names for this matching rule use, or an empty list if
   *          there are no user-defined names.
   */
  List<String> getNames();



  /**
   * Indicates whether the provided string is equal to any of the defined names
   * for this matching rule use.
   *
   * @param  name  The name for which to make the determination.
   *
   * @return  {@code true} if the provided string matches one of the names for
   *          this matching rule use, or {@code false} if not.
   */
  boolean hasName(final String name);



  /**
   * Retrieves the matching rule for this matching rule use.
   *
   * @return  The matching rule for this matching rule use.
   */
  MatchingRule getMatchingRule();



  /**
   * Retrieves the description for this DIT content rule, if any.
   *
   * @return  The description for this DIT content rule, or {@code null} if it
   *          does not have a description.
   */
  String getDescription();



  /**
   * Retrieves the set of attributes that are associated with this matching rule
   * use.
   *
   * @return  The set of attributes that are associated with this matching rule
   *          use.
   */
  Set<AttributeType> getAttributes();



  /**
   * Indicates whether this matching rule use applies to the provided attribute
   * type.
   *
   * @param  t  The attribute type for which to make the determination.
   *
   * @return  {@code true} if this matching rule use applies to the provided
   *          attribute type, or {@code false} if not.
   */
  boolean appliesToAttribute(final AttributeType t);



  /**
   * Indicates whether this matching rule use is declared obsolete in the server
   * schema.
   *
   * @return  {@code true} if this matching rule use is declared obsolete in the
   *          server schema, or {@code false} if not.
   */
  boolean isObsolete();



  /**
   * Retrieves a map of all defined extensions for this matching rule use.
   *
   * @return  A map of all defined extensions for this matching rule use.
   */
  Map<String,List<String>> getExtensions();



  /**
   * Retrieves the name of the schema file in which this matching rule use is
   * defined.
   *
   * @return  The name of the schema file in which this matching rule use is
   *          defined.
   */
  String getSchemaFileName();



  /**
   * Indicates whether the provided object is equal to this matching rule use.
   *
   * @param  o  The object for which to make the determination.
   *
   * @return  {@code true} if the provided object is equal to this matching rule
   *          use, or {@code false} if not.
   */
  boolean equals(final Object o);



  /**
   * Retrieves a hash code for this matching rule use.
   *
   * @return  A hash code for this matching rule use.
   */
  int hashCode();



  /**
   * Retrieves a string representation of this matching rule use definition.
   *
   * @return  A string representation of this matching rule use definition.
   */
  String toString();
}
