/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.schema;



import java.util.List;
import java.util.Map;
import java.util.Set;

import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be used to obtain
 * information about a name form defined in the server schema.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface NameForm
{
  /**
   * Retrieves the numeric OID for this name form.
   *
   * @return  The numeric OID for this name form.
   */
  String getOID();



  /**
   * Retrieves the list of names for this name form, if any.
   *
   * @return  The list of names for this name form, or an empty list if
   *          there are no user-defined names.
   */
  List<String> getNames();



  /**
   * Retrieves the primary name for this name form, or the numeric OID if no
   * names are defined.
   *
   * @return The primary name or OID for this name form.
   */
  String getNameOrOID();



  /**
   * Indicates whether the provided string is equal to the OID or any of the
   * defined names for this name form.
   *
   * @param  name  The name for which to make the determination.
   *
   * @return  {@code true} if the provided string matches the OID or one of the
   *          names for this name form, or {@code false} if not.
   */
  boolean hasNameOrOID(final String name);



  /**
   * Retrieves the description for this name form, if any.
   *
   * @return  The description for this name form, or {@code null} if it does not
   *          have a description.
   */
  String getDescription();



  /**
   * Retrieves the structural object class for this name form.
   *
   * @return  The structural object class for this name form.
   */
  ObjectClass getStructuralClass();



  /**
   * Retrieves the set of required attributes for this name form.
   *
   * @return  The set of required attributes for this name form.
   */
  Set<AttributeType> getRequiredAttributes();



  /**
   * Indicates whether the provided attribute type is required by this name
   * form.
   *
   * @param  t  The attribute type for which to make the determination.
   *
   * @return  {@code true} if the provided attribute type is required by this
   *          name form, or {@code false} if not.
   */
  boolean isRequired(final AttributeType t);



  /**
   * Retrieves the set of optional attributes for this name form, if any.
   *
   * @return  The set of optional attributes for this name form, or an empty set
   *          if there are no optional attributes.
   */
  Set<AttributeType> getOptionalAttributes();



  /**
   * Indicates whether the provided attribute type is optional for this name
   * form.
   *
   * @param  t  The attribute type for which to make the determination.
   *
   * @return  {@code true} if the provided attribute type is optional for this
   *          name form, or {@code false} if not.
   */
  boolean isOptional(final AttributeType t);



  /**
   * Indicates whether the provided attribute type is allowed for use with this
   * name form as either a required or optional type.
   *
   * @param  t  The attribute type for which to make the determination.
   *
   * @return  {@code true} if the provided attribute type is allowed for use
   *          with this name form, or {@code false} if not.
   */
  boolean isRequiredOrOptional(final AttributeType t);



  /**
   * Indicates whether this name form is declared obsolete in the server schema.
   *
   * @return  {@code true} if this name form is declared obsolete in the server
   *          schema, or {@code false} if not.
   */
  boolean isObsolete();



  /**
   * Retrieves a map of all defined extensions for this name form.
   *
   * @return  A map of all defined extensions for this name form.
   */
  Map<String,List<String>> getExtensions();



  /**
   * Retrieves the name of the schema file in which this name form is defined.
   *
   * @return  The name of the schema file in which this name form is defined.
   */
  String getSchemaFileName();



  /**
   * Indicates whether the provided object is equal to this name form.
   *
   * @param  o  The object for which to make the determination.
   *
   * @return  {@code true} if the provided object is equal to this name form, or
   *          {@code false} if not.
   */
  boolean equals(final Object o);



  /**
   * Retrieves a hash code for this name form.
   *
   * @return  A hash code for this name form.
   */
  int hashCode();



  /**
   * Retrieves a string representation of this name form definition.
   *
   * @return  A string representation of this name form definition.
   */
  String toString();
}
