/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.schema;



import java.util.List;
import java.util.Map;
import java.util.Set;

import com.unboundid.ldap.sdk.schema.ObjectClassType;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be used to obtain
 * information about an object class defined in the server schema.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface ObjectClass
{
  /**
   * Retrieves the numeric OID for this object class.
   *
   * @return  The numeric OID for this object class.
   */
  String getOID();



  /**
   * Retrieves the list of names for this object class, if any.
   *
   * @return  The list of names for this object class, or an empty list if
   *          there are no user-defined names.
   */
  List<String> getNames();



  /**
   * Retrieves the primary name for this object class, or the numeric OID if
   * no names are defined.
   *
   * @return The primary name or OID for this object class.
   */
  String getNameOrOID();



  /**
   * Indicates whether the provided string is equal to the OID or any of the
   * defined names for this object class.
   *
   * @param  name  The name for which to make the determination.
   *
   * @return  {@code true} if the provided string matches the OID or one of the
   *          names for this object class, or {@code false} if not.
   */
  boolean hasNameOrOID(final String name);



  /**
   * Retrieves the description for this object class, if any.
   *
   * @return  The description for this object class, or {@code null} if it does
   *          not have a description.
   */
  String getDescription();



  /**
   * Retrieves the immediate superior classes for this object class, if any.
   *
   * @return  The immediate superior classes for this object class, or an empty
   *          list if this object class does not have any superior classes.
   */
  List<ObjectClass> getSuperiorClasses();



  /**
   * Retrieves the set of required attributes for this object class.  This set
   * will only include the required attributes defined for this object class
   * itself and not those of its superior classes.
   *
   * @return  The set of required attributes for this object class, or an empty
   *          list if it does not have any required attributes.
   */
  Set<AttributeType> getRequiredAttributes();



  /**
   * Retrieves the set of required attributes for this object class and all of
   * its superior classes.
   *
   * @return  The set of required attributes for this object class and all of
   *          its superior classes, or an empty list if it does not have any
   *          required attributes.
   */
  Set<AttributeType> getRequiredAttributeChain();



  /**
   * Indicates whether the provided attribute type is required for use with this
   * object class or any of its superior classes.
   *
   * @param  t  The attribute type for which to make the definition.
   *
   * @return  {@code true} if the provided attribute type is required for use
   *          with this object class, or {@code false} if not.
   */
  boolean isRequired(final AttributeType t);



  /**
   * Retrieves the set of optional attributes for this object class.  This set
   * will only include the optional attributes defined for this object class
   * itself and not those of its superior classes.
   *
   * @return  The set of optional attributes for this object class, or an empty
   *          list if it does not have any optional attributes.
   */
  Set<AttributeType> getOptionalAttributes();



  /**
   * Retrieves the set of optional attributes for this object class and all of
   * its superior classes.
   *
   * @return  The set of optional attributes for this object class and all of
   *          its superior classes, or an empty list if it does not have any
   *          optional attributes.
   */
  Set<AttributeType> getOptionalAttributeChain();



  /**
   * Indicates whether the provided attribute type is optional for use with this
   * object class or any of its superior classes.
   *
   * @param  t  The attribute type for which to make the definition.
   *
   * @return  {@code true} if the provided attribute type is optional for use
   *          with this object class, or {@code false} if not.
   */
  boolean isOptional(final AttributeType t);



  /**
   * Indicates whether the provided attribute type is allowed for use with this
   * object class or any of its superior classes, either as a required or
   * optional type.
   *
   * @param  t  The attribute type for which to make the definition.
   *
   * @return  {@code true} if the provided attribute type is allowed for use
   *          with this object class, or {@code false} if not.
   */
  boolean isRequiredOrOptional(final AttributeType t);



  /**
   * Retrieves the object class type for this object class.
   *
   * @return  The object class type for this object class.
   */
  ObjectClassType getObjectClassType();



  /**
   * Indicates whether this object class is declared obsolete in the server
   * schema.
   *
   * @return  {@code true} if this object class is declared obsolete in the
   *          server schema, or {@code false} if not.
   */
  boolean isObsolete();



  /**
   * Retrieves a map of all defined extensions for this object class.
   *
   * @return  A map of all defined extensions for this object class.
   */
  Map<String,List<String>> getExtensions();



  /**
   * Retrieves the name of the schema file in which this object class is
   * defined.
   *
   * @return  The name of the schema file in which this object class is defined.
   */
  String getSchemaFileName();



  /**
   * Indicates whether the provided object is equal to this object class.
   *
   * @param  o  The object for which to make the determination.
   *
   * @return  {@code true} if the provided object is equal to this object class,
   *          or {@code false} if not.
   */
  boolean equals(final Object o);



  /**
   * Retrieves a hash code for this object class.
   *
   * @return  A hash code for this object class.
   */
  int hashCode();



  /**
   * Retrieves a string representation of this object class definition.
   *
   * @return  A string representation of this object class definition.
   */
  String toString();
}
