/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.schema;



import java.util.Map;

import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface provides a number of methods that may be used to interact with
 * the server schema.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface Schema
{
  /**
   * Retrieves the set of attribute types defined in the server schema, mapped
   * from lowercase name or OID to the attribute type definition.  Each
   * attribute type definition may be included in the map multiple times with
   * each of its names as well as its OID.
   *
   * @return  The set of attribute types defined in the server schema, mapped
   *          from lowercase name or OID to the attribute type definition.
   */
  Map<String,AttributeType> getAttributeTypesByName();



  /**
   * Retrieves the specified attribute type from the server schema, optionally
   * creating a new attribute type with the specified name if none is defined.
   *
   * @param  name           The name or OID of the attribute type to retrieve.
   * @param  returnDefault  Indicates whether to create a new attribute type
   *                        with the given name and a default settings if no
   *                        such attribute is defined in the server schema.
   *
   * @return  The requested attribute type, or {@code null} if no such attribute
   *          is defined in the schema and a default type should not be
   *          returned.
   */
  AttributeType getAttributeType(final String name,
                                 final boolean returnDefault);



  /**
   * Retrieves the set of object classes defined in the server schema, mapped
   * from lowercase name or OID to the object class definition.  Each object
   * class definition may be included in the map multiple times with each of its
   * names as well as its OID.
   *
   * @return  The set of object classes defined in the server schema, mapped
   *          from lowercase name or OID to the object class definition.
   */
  Map<String,ObjectClass> getObjectClassesByName();



  /**
   * Retrieves the specified object class from the server schema, optionally
   * creating a new class with the specified name if none is defined.
   *
   * @param  name           The name or OID of the object class to retrieve.
   * @param  returnDefault  Indicates whether to create a new object class
   *                        with the given name and a default settings if no
   *                        such object class is defined in the server schema.
   *
   * @return  The requested object class, or {@code null} if no such object
   *          class is defined in the schema and a default class should not be
   *          returned.
   */
  ObjectClass getObjectClass(final String name, final boolean returnDefault);



  /**
   * Retrieves the set of attribute syntaxes defined in the schema, mapped from
   * OID to the corresponding attribute syntax.
   *
   * @return  The set of attribute syntaxes defined in the server schema, mapped
   *          from OID to the attribute syntax definition.
   */
  Map<String,AttributeSyntax> getAttributeSyntaxesByOID();



  /**
   * Retrieves the attribute syntax with the specified OID, if available.
   *
   * @param  oid  The OID of the attribute syntax to retrieve.
   *
   * @return  The requested attribute syntax, or {@code null} if there is no
   *          such syntax defined in the server.
   */
  AttributeSyntax getAttributeSyntax(final String oid);



  /**
   * Retrieves the set of matching rules defined in the server schema, mapped
   * from lowercase name or OID to the matching rule definition.  Each
   * matching rule definition may be included in the map multiple times with
   * each of its names as well as its OID.
   *
   * @return  The set of matching rules defined in the server schema, mapped
   *          from lowercase name or OID to the matching rule definition.
   */
  Map<String,MatchingRule> getMatchingRulesByName();



  /**
   * Retrieves the specified matching rule from the server schema.
   *
   * @param  name  The name or OID of the matching rule to retrieve.
   *
   * @return  The requested matching rule, or {@code null} if no such matching
   *          rule is defined in the schema.
   */
  MatchingRule getMatchingRule(final String name);



  /**
   * Retrieves the set of name forms defined in the server schema, mapped from
   * structural object class to the corresponding name form definition.
   *
   * @return  The set of name forms defined in the server schema, mapped from
   *          structural object class to the corresponding name form definition.
   */
  Map<ObjectClass,NameForm> getNameFormsByStructuralClass();



  /**
   * Retrieves the specified name form from the server schema.
   *
   * @param  name  The name or OID of the name form to retrieve.
   *
   * @return  The requested name form, or {@code null} if there is no name form
   *          with the specified name or OID.
   */
  NameForm getNameForm(final String name);



  /**
   * Retrieves the specified name form from the server schema.
   *
   * @param  structuralClass  The structural object class for which to retrieve
   *                          the corresponding name form.
   *
   * @return  The requested name form, or {@code null} if there is no name form
   *          associated with the specified structural object class.
   */
  NameForm getNameForm(final ObjectClass structuralClass);



  /**
   * Retrieves the set of DIT content rules defined in the server schema, mapped
   * from structural object class to the corresponding DIT structure rule
   * definition.
   *
   * @return  The set of DIT content rules defined in the server schema, mapped
   *          from structural object class to the corresponding DIT content
   *          rule definition.
   */
  Map<ObjectClass,DITContentRule> getDITContentRulesByStructuralClass();



  /**
   * Retrieves the specified DIT content rule from the server schema.
   *
   * @param  structuralClass  The structural object class for which to retrieve
   *                          the corresponding DIT content rule.
   *
   * @return  The requested DIT content rule, or {@code null} if no such DIT
   *          content rule is defined in the server schema.
   */
  DITContentRule getDITContentRule(final ObjectClass structuralClass);



  /**
   * Retrieves the set of DIT structure rules defined in the server schema,
   * mapped from rule ID to the corresponding DIT structure rule definition.
   *
   * @return  The set of DIT structure rules defined in the server schema,
   *          mapped from rule ID to the corresponding DIT structure rule
   *          definition.
   */
  Map<Integer,DITStructureRule> getDITStructureRulesByRuleID();



  /**
   * Retrieves the specified DIT structure rule from the server schema.
   *
   * @param  ruleID  The rule ID for the DIT structure rule to retrieve.
   *
   * @return  The requested DIT structure rule, or {@code null} if no such DIT
   *          structure rule is defined in the server schema.
   */
  DITStructureRule getDITStructureRule(final int ruleID);



  /**
   * Retrieves the specified DIT structure rule from the server schema.
   *
   * @param  nameForm  The name form for which to retrieve the corresponding DIT
   *                   structure rule.
   *
   * @return  The requested DIT structure rule, or {@code null} if no such DIT
   *          structure rule is defined in the server schema.
   */
  DITStructureRule getDITStructureRule(final NameForm nameForm);



  /**
   * Retrieves the set of matching rule uses defined in the server schema,
   * mapped from matching rule to its corresponding matching rule use.
   *
   * @return  The set of matching rule uses defined in the server schema, mapped
   *          from matching rule to its corresponding matching rule use.
   */
  Map<MatchingRule,MatchingRuleUse> getMatchingRuleUsesByMatchingRule();



  /**
   * Retrieves the specified matching rule use from the server schema.
   *
   * @param  matchingRule  The matching rule for which to retrieve the
   *                       corresponding matching rule use.
   *
   * @return  The requested matching rule use, or {@code null} if there is no
   *          matching rule use associated with the provided matching rule.
   */
  MatchingRuleUse getMatchingRuleUse(final MatchingRule matchingRule);
}
