/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.schema;



import java.util.List;

import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.util.ByteString;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface provides an API for interacting with substring matching rules,
 * which can be used to determine whether a given value matches a provided
 * substring assertion.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface SubstringMatchingRule
       extends MatchingRule
{
  /**
   * Normalizes the provided value fragment into a form that can be used to
   * efficiently compare it in a substring assertion.
   *
   * @param  substring  The substring value to be normalized.
   *
   * @return  The normalized form of the provided substring.
   *
   * @throws  LDAPException  If a problem is encountered while attempting to
   *                         normalize the provided substring.
   */
  ByteString normalizeSubstring(final ByteString substring)
             throws LDAPException;



  /**
   * Normalizes the provided value fragment into a form that can be used to
   * efficiently compare it in a substring assertion.
   *
   * @param  type       The associated attribute type.
   * @param  substring  The substring value to be normalized.
   *
   * @return  The normalized form of the provided substring.
   *
   * @throws  LDAPException  If a problem is encountered while attempting to
   *                         normalize the provided substring.
   */
  ByteString normalizeSubstring(final AttributeType type,
                                final ByteString substring)
             throws LDAPException;



  /**
   * Indicates whether the provided value matches the given substring assertion.
   *
   * @param  normValue       The normalized form of the value to be compared.
   * @param  normSubInitial  The normalized form of the subInitial element for
   *                         the substring assertion.  It may be {@code null} if
   *                         there is no subInitial element.
   * @param  normSubAny      The normalized forms of the subAny elements for the
   *                         substring assertion.  It may be empty or
   *                         {@code null} if there is no subInitial element.
   * @param  normSubFinal    The normalized form of the subFinal element for
   *                         the substring assertion.  It may be {@code null} if
   *                         there is no subFinal element.
   *
   * @return  {@code true} if the provided value matches the given substring
   *          assertion, or {@code false} if not.
   */
  boolean valueMatchesSubstring(final ByteString normValue,
                                final ByteString normSubInitial,
                                final List<ByteString> normSubAny,
                                final ByteString normSubFinal);
}
