/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2011-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.scripting;



import java.util.List;

import com.unboundid.directory.sdk.broker.internal.BrokerExtension;
import com.unboundid.directory.sdk.common.config.FileBasedErrorLoggerConfig;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.types.LogCategory;
import com.unboundid.directory.sdk.common.types.LogSeverity;
import com.unboundid.directory.sdk.common.types.ServerContext;
import com.unboundid.directory.sdk.metrics.internal.MetricsEngineExtension;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that may be used to create a specific type of
 * scripted error logger which is intended to write log messages to text files.
 * This is a convenience for developers which wish to create custom error
 * loggers that write to text files and provides support for a wide range of
 * functionality including high-performance and highly-concurrent logging.  All
 * of the options available to {@link ScriptedErrorLogger} implementations are
 * available for file-based error loggers, as well as options for indicating
 * the log file path, the rotation and retention policies, whether to buffer the
 * output, etc.
 * <BR><BR>
 * Note that scripted file-based error loggers will automatically be registered
 * within the server as disk space consumers, so there is no need to implement
 * the {@link com.unboundid.directory.sdk.common.api.DiskSpaceConsumer}
 * interface.  Also note that configuration change related to the log file
 * (e.g., the log file path, buffer size, queue size, etc.) will also
 * automatically be handled by the server, so subclasses only need to be
 * concerned about changes to the custom arguments they define.
 * <BR>
 * <H2>Configuring File-Based Error Loggers</H2>
 * In order to configure an error logger created using this API, use a command
 * like:
 * <PRE>
 *      dsconfig create-log-publisher \
 *           --publisher-name "<I>{logger-name}</I>" \
 *           --type groovy-scripted-file-based-error \
 *           --set enabled:true \
 *           --set "log-file:<I>{path}</I>" \
 *           --set "rotation-policy:<I>{rotation-policy-name}</I>" \
 *           --set "retention-policy:<I>{retention-policy-name}</I>" \
 *           --set "script-class:<I>{class-name}</I>" \
 *           --set "script-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{logger-name}</I>" is the name to use for the error logger
 * instance, "<I>{path}</I>" is the path to the log file to be written,
 * "<I>{rotation-policy-name}</I>" is the name of the log rotation policy to use
 * for the log file, "<I>{retention-policy-name}</I>" is the name of the log
 * retention policy to use for the log file, "<I>{class-name}</I>" is the
 * fully-qualified name of the Groovy class written using this API, and
 * "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the logger.  If multiple arguments should be provided to the
 * logger, then the "<CODE>--set script-argument:<I>{name=value}</I></CODE>"
 * option should be provided multiple times.  It is also possible to specify
 * multiple log rotation and/or retention policies if desired.
 *
 * @see  com.unboundid.directory.sdk.common.api.ErrorLogger
 * @see  com.unboundid.directory.sdk.common.api.FileBasedErrorLogger
 * @see  ScriptedErrorLogger
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=false)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@MetricsEngineExtension()
@BrokerExtension()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class ScriptedFileBasedErrorLogger
       implements Reconfigurable<FileBasedErrorLoggerConfig>
{
  /**
   * Creates a new instance of this file-based error logger.  All file-based
   * error logger implementations must include a default constructor, but any
   * initialization should generally be done in the
   * {@code initializeErrorLogger} method.
   */
  public ScriptedFileBasedErrorLogger()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this file-based error logger.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this file-based error
   *                        logger.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this file-based error logger.
   *
   * @throws  LDAPException  If a problem occurs while initializing this
   *                         file-based error logger.
   */
  public void initializeErrorLogger(final ServerContext serverContext,
                                    final FileBasedErrorLoggerConfig config,
                                    final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * Performs any cleanup which may be necessary when this file-based error
   * logger is to be taken out of service.
   */
  public void finalizeErrorLogger()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final FileBasedErrorLoggerConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final FileBasedErrorLoggerConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // If there are any custom arguments, then add an admin action message
    // indicating that the extension needs to be restarted for any changes to
    // take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "If any extension-argument values have been altered, then " +
                "those new values have not actually been applied.  The new " +
                "configuration for those arguments will not take effect " +
                "until this file-based error logger is disabled and " +
                "re-enabled, or until the server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Records information about the provided message, if appropriate.
   *
   * @param  category   The category for the message to be logged.
   * @param  severity   The severity for the message to be logged.
   * @param  messageID  The unique identifier with which the message text is
   *                    associated.
   * @param  message    The message to be logged.
   *
   * @return  The content of the log message that should be written.  It may be
   *          {@code null} or empty if no message should be written.  It may
   *          optionally include line breaks if the log message should span
   *          multiple lines.
   */
  public abstract CharSequence logError(final LogCategory category,
                                        final LogSeverity severity,
                                        final long messageID,
                                        final String message);
}
