/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;



import com.unboundid.ldap.sdk.IntermediateResponse;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be used to obtain
 * information about an operation that is actively being processed and for which
 * the final response has not yet been sent.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_NOT_THREADSAFE)
public interface ActiveOperationContext
       extends OperationContext
{
  /**
   * Attempts to send an intermediate response message to the client.
   *
   * @param  r  The intermediate response to be sent.  It must not be
   *            {@code null}.
   *
   * @throws  LDAPException  If a problem occurs while attempting to send the
   *                         intermediate response message and processing on the
   *                         associated operation should be aborted.
   */
  void sendIntermediateResponse(final IntermediateResponse r)
       throws LDAPException;



  /**
   * Sets an attachment for the associated operation.
   *
   * @param  name   The name of the attachment to set.  It must not be
   *                {@code null}.
   * @param  value  The value to set for the attachment.  It may be {@code null}
   *                if any existing attachment with the specified name should be
   *                removed.
   */
  void setAttachment(final String name, final Object value);



  /**
   * Indicates whether the requester has the specified privilege, optionally
   * including it in the set of used privileges for the operation.  The set of
   * defined privileges may be found in the privilege-list.html and
   * privilege-list.csv files in the server docs directory.
   *
   * @param  privilegeName  The name of the privilege for which to make the
   *                        determination.  It must not be {@code null}.
   * @param  markUsed       Indicates whether to include the specified privilege
   *                        in the set of privileges that were used in the
   *                        course of processing the operation.  This is only
   *                        applicable if the requester has the specified
   *                        privilege.
   *
   * @return  {@code true} if the requester has the specified privilege, or
   *          {@code false} if not.
   *
   * @throws  LDAPException  If the specified privilege is not defined in the
   *                         server, or if a problem is encountered while trying
   *                         to make the determination.
   */
  boolean hasPrivilege(final String privilegeName, final boolean markUsed)
          throws LDAPException;



  /**
   * Ensures that the requester has the specified privilege.  If the requester
   * has the specified privilege, then the set of privileges used in the course
   * of processing the operation will be updated to include the specified
   * privilege.  If the specifies privilege is defined in the server but the
   * requested user does not have it, then it will be added to the set of
   * missing privileges for the operation.
   * <BR><BR>
   * The set of defined privileges may be found in the privilege-list.html and
   * privilege-list.csv files in the server docs directory.
   *
   * @param  privilegeName  The name of the privilege to require.  It must not
   *                        be {@code null}.
   *
   * @throws  LDAPException  If the specified privilege is not defined in the
   *                         server, if the requester does not have that
   *                         privilege, or a problem is encountered while trying
   *                         to make the determination.
   */
  void requirePrivilege(final String privilegeName)
       throws LDAPException;
}
