/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2014-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;

import java.util.HashMap;
import java.util.Map;

/**
 * Enumerates severity types for alarm conditions from least severe to most
 * severe.
 */
public enum AlarmSeverity {
  /**
   * Indicates that an alarm has been cleared and the alarming condition
   * has returned to normal.
   * <p>
   * It should be noted that the x733 specification does not define a normal
   * severity but rather lack of an alarm indicates normality.  We use this
   * severity for normalized alarms that are retained in the alarm backend as
   * well as for indicating to the SNMP alert handler that an alarm has been
   * cleared, either because its condition has abated or if the server is
   * configured to send interim clear traps between severity changes.  The
   * latter use case does not imply that an alarm has returned to normal.
   */
  NORMAL,

  /**
   * Indicates the existence of a potential or impending service
   * affecting fault, before any significant effects have been felt.
   * Action should be taken to further diagnose (if necessary) and
   * correct the problem in order to prevent it from becoming a more
   * serious service affecting fault.
   */
  WARNING,

  /**
   * Indicates the existence of a non-service affecting fault condition
   * and that corrective action should be taken in order to prevent a
   * more serious (for example, service affecting) fault. Such a severity
   * can be reported, for example, when the detected alarm condition is
   * not currently degrading the capacity of the managed object.
   */
  MINOR,

  /**
   * Indicates that a service affecting condition has developed and an
   * urgent corrective action is required. Such a severity can be
   * reported, for example, when there is a severe degradation in the
   * capability of the managed object and its full capability must be
   * restored.
   */
  MAJOR,

  /**
   * Indicates that a service affecting condition has occurred and an
   * immediate corrective action is required. Such a severity can be
   * reported, for example, when a managed object becomes totally out
   * of service and its capability must be restored.
   */
  CRITICAL;

  private static Map<String, AlarmSeverity> displayNameLookup;
  static {
    displayNameLookup = new HashMap<String, AlarmSeverity>();
    for (final AlarmSeverity severity : AlarmSeverity.values()) {
      displayNameLookup.put(
          severity.name().toLowerCase(), severity);
    }
  }

  /**
   * Returns the human-friendly name of this severity.
   *
   * @return name of this severity.
   */
  public String getDisplayName() {
    return name().toLowerCase();
  }

  /**
   * Get the alarm severity for the given display name.
   *
   * @param displayName  Display name for which alarm severity is requested.
   * @return  The alarm severity.
   */
  public static AlarmSeverity forDisplayName(final String displayName) {
    return displayNameLookup.get(displayName.toLowerCase());
  }
}
