/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;



import java.util.Date;
import java.util.Map;

import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be used to interact with an
 * alert notification.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface AlertNotification
{
  /**
   * Retrieves the unique ID assigned to this alert notification.
   *
   * @return  The unique ID assigned to this alert notification.
   */
  String getAlertID();



  /**
   * Retrieves the time that this alert notification was generated.
   *
   * @return  The time that this alert notification was generated.
   */
  Date getAlertTime();



  /**
   * Retrieves the name of the class that generated this alert notification.
   *
   * @return  The name of the class that generated this alert notification.
   */
  String getAlertGeneratorClassName();



  /**
   * Retrieves the name of the alert type for this alert notification.
   *
   * @return  The name of the alert type for this alert notification.
   */
  String getAlertTypeName();



  /**
   * Retrieves a short summary of the alert type for this alert notification.
   * It will generally use headline-style capitalization.
   *
   * @return  A short summary of the alert type for this alert notification.
   */
  String getAlertTypeSummary();



  /**
   * Retrieves a description of the alert type for this alert notification.  It
   * may be longer than the alert type summary and will use a more prose-style
   * capitalization.
   *
   * @return  A description of the alert type for this alert notification.
   */
  String getAlertTypeDescription();



  /**
   * Retrieves the OID of the alert type for this alert notification.
   *
   * @return  The OID of the alert type for this alert notification.
   */
  String getAlertTypeOID();



  /**
   * Retrieves the severity for this alert notification.
   *
   * @return  The severity for this alert notification.
   */
  AlertSeverity getAlertSeverity();



  /**
   * Retrieves the message for this alert notification, which may contain
   * specific information about the error, warning, or event that triggered the
   * alert.
   *
   * @return  The message for this alert notification.
   */
  String getAlertMessage();


  /**
   * Retrieves a map of objects containing additional information for this alert
   * notification.
   *
   * @return  Map of objects for this alert notification.
   */
  Map<Object,Object> getAdditionalInformation();



  /**
   * Retrieves a string representation of this alert notification.
   *
   * @return  A string representation of this alert notification.
   */
  String toString();
}
