/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;



import com.unboundid.util.StaticUtils;



/**
 * This enumeration defines the severities that may be used for alert types.
 */
public enum AlertSeverity
{
  /**
   * Indicates that the associated alert type is used for informational
   * messages.
   */
  INFO("info"),



  /**
   * Indicates that the associated alert type is used for warning messages.
   */
  WARNING("warning"),



  /**
   * Indicates that the associated alert type is used for error messages.
   */
  ERROR("error"),



  /**
   * Indicates that the associated alert type is used for fatal alerts that
   * cause the server to shut down or lose significant functionality.
   */
  FATAL("fatal");



  // The name for this alert severity.
  private final String name;



  /**
   * Creates a new alert severity with the specified name.
   *
   * @param  name  The name for this alert severity.
   */
  AlertSeverity(final String name)
  {
    this.name = name;
  }



  /**
   * Retrieves the name of this alert severity.
   *
   * @return  The name of this alert severity.
   */
  public String getName()
  {
    return name;
  }



  /**
   * Retrieves the alert severity with the specified name.
   *
   * @param  name  The name of the alert severity to retrieve.
   *
   * @return  The requested alert severity, or {@code null} if there is no such
   *          severity.
   */
  public static AlertSeverity forName(final String name)
  {
    final String lowerName = StaticUtils.toLowerCase(name);

    if (lowerName.equals("info"))
    {
      return INFO;
    }
    else if (lowerName.equals("warning"))
    {
      return WARNING;
    }
    else if (lowerName.equals("error"))
    {
      return ERROR;
    }
    else if (lowerName.equals("fatal"))
    {
      return FATAL;
    }

    return null;
  }
}
