/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;



import java.net.InetAddress;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.unboundid.ldap.sdk.DN;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be used to obtain
 * information about a client connection that has been established to the
 * server.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_NOT_THREADSAFE)
public interface ClientContext
{
  /**
   * Retrieves the identifier that has been assigned to the associated client
   * connection.
   *
   * @return  The identifier that has been assigned to the associated client
   *          connection.
   */
  long getConnectionID();



  /**
   * Returns a {@code List} containing the names of all the Connection
   * Criteria that match this connection. These are the configuration names
   * (e.g. the RDNs, not the full DNs) of the Connection Criteria.
   *
   * @return a list of connection criteria names.
   */
  List<String> getMatchedConnectionCriteria();



  /**
   * Determines whether this {@code ClientContext} matches the given Connection
   * Criteria.
   *
   * @param criteriaName the name (not the DN) of the Connection Criteria to
   *                     check against.
   * @return true if this {@code ClientContext} matches the specified
   *         Connection Criteria, false otherwise.
   */
  boolean matchesConnectionCriteria(final String criteriaName);



  /**
   * Indicates whether this represents an internal client connection.
   *
   * @return  {@code true} if this represents an internal client connection, or
   *          {@code false} if it is from an external client.
   */
  boolean isInternal();



  /**
   * Retrieves an internal connection that is authenticated as a root user
   * that is not subject to access control.  It may optionally use the client
   * connection policy from the associated client connection.
   *
   * @param  usePolicyFromConnection  If {@code true}, the internal connection
   *                                  will use the same client connection policy
   *                                  as the associated client connection.  If
   *                                  {@code false}, the internal connection
   *                                  will use the server's default client
   *                                  connection policy for internal
   *                                  connections.
   *
   * @return  An internal connection that is authenticated as a root user.
   */
  InternalConnection getInternalRootConnection(
                          final boolean usePolicyFromConnection);



  /**
   * Retrieves an internal connection that is authenticated as the same user
   * as the associated client connection.  It may optionally use the client
   * connection policy from the associated client connection.
   *
   * @param  usePolicyFromConnection  If {@code true}, the internal connection
   *                                  will use the same client connection policy
   *                                  as the associated client connection.  If
   *                                  {@code false}, the internal connection
   *                                  will use the server's default client
   *                                  connection policy for internal
   *                                  connections.
   *
   * @return  An internal connection that is authenticated as the same user as
   *          the associated client connection.
   *
   * @throws  LDAPException  If a problem occurs while attempting to obtain or
   *                         authenticate the connection.
   */
  InternalConnection getInternalUserConnection(
                          final boolean usePolicyFromConnection)
       throws LDAPException;



  /**
   * Retrieves an internal connection that is authenticated as the specified
   * user.  Operations on the connection may be subject to access control based
   * on the privileges associated with the specified user.  It may optionally
   * use the client connection policy from the associated client connection.
   *
   * @param  dn                       The DN of the user as whom the connection
   *                                  should be authenticated.  It may be
   *                                  {@code null} or empty if the connection
   *                                  should be unauthenticated.
   * @param  usePolicyFromConnection  If {@code true}, the internal connection
   *                                  will use the same client connection policy
   *                                  as the associated client connection.  If
   *                                  {@code false}, the internal connection
   *                                  will use the server's default client
   *                                  connection policy for internal
   *                                  connections.
   *
   * @return  An internal connection that is authenticated as the specified
   *          user.
   *
   * @throws  LDAPException  If a problem occurs while attempting to
   *                         authenticate as the specified user.
   */
  InternalConnection getInternalConnection(final String dn,
                          final boolean usePolicyFromConnection)
       throws LDAPException;



  /**
   * Indicates whether the client is communicating with the server in a secure
   * manner.
   *
   * @return  {@code true} if the client is communicating with the server in a
   *          secure manner, or {@code false} if not.
   */
  boolean isSecure();



  /**
   * Retrieves the name of the protocol that the client is using to communicate
   * with the server.
   *
   * @return  The name of the protocol that the client is using to communicate
   *          with the server.
   */
  String getProtocol();



  /**
   * Retrieves the time that the connection was established.  The value returned
   * will be an offset in milliseconds since 12:00 a.m. on January 1, 1970.
   *
   * @return  The time that the connection was established.
   */
  long getConnectTime();



  /**
   * Retrieves an {@code InetAddress} representing the address of the client
   * system, if available.
   *
   * @return  An {@code InetAddress} representing the address of the client
   *          system, or {@code null} if that is not available or applicable for
   *          the associated client connection.
   */
  InetAddress getClientInetAddress();



  /**
   * Retrieves an {@code InetAddress} representing the address on the server to
   * which the client established the connection, if available.
   *
   * @return  The address on the server to which the client established the
   *          connection, or {@code null} if that is not available or
   *          applicable.
   */
  InetAddress getServerInetAddress();



  /**
   * Indicates whether the client has authenticated to the server.
   *
   * @return  {@code true} if the client has authenticated to the server, or
   *          {@code false} if not.
   */
  boolean isAuthenticated();



  /**
   * Retrieves information about the authentication state of the client
   * connection.
   *
   * @return  Information about the authentication state of the client
   *          connection.
   */
  AuthInfo getAuthInfo();



  /**
   * Indicates whether the currently-authenticated user is a member of the
   * specified group.  This will consider both direct memberships (in which the
   * user is explicitly named as a member of the target group) and indirect
   * memberships (in which the user is a member of the group by virtue of
   * matching dynamic group criteria or by membership in a nested group).
   *
   * @param  groupDN    The DN of the group for which to make the determination.
   *                    It must not be {@code null}.
   * @param  operation  The operation currently being processed.  It may be
   *                    {@code null} if no operation is available.
   *
   * @return  {@code true} if the authenticated user is a member of the
   *          specified group, or {@code false} if not.
   *
   * @throws  LDAPException  If a problem is encountered while attempting to
   *                         make the determination.
   */
  boolean isMemberOf(final String groupDN, final OperationContext operation)
          throws LDAPException;



  /**
   * Retrieves the groups in which the currently-authenticated user is a member,
   * indexed by group DN.
   *
   * @param  operation             The operation currently being processed.  It
   *                               may be {@code null} if no operation is
   *                               available.
   * @param  directMembershipOnly  Indicates whether to only consider groups in
   *                               which the user is directly named as a member.
   *                               If this parameter is {@code true}, then only
   *                               static groups that directly contain the
   *                               authenticated user will be included.  If this
   *                               parameter is {@code false}, then the set of
   *                               groups returned will also include dynamic
   *                               groups in which the user's entry matches the
   *                               membership criteria, as well as static groups
   *                               in which the user is a nested member.
   *
   * @return  The groups in which the currently-authenticated user is a member,
   *          or an empty map if the client connection is not authenticated or
   *          if the authenticated user is not a member of any groups.
   *
   * @throws  LDAPException  If a problem is encountered while attempting to
   *                         determine the set of groups in which the
   *                         authenticated user is a member.
   */
  Map<DN,Group> getGroups(final OperationContext operation,
                          final boolean directMembershipOnly)
                throws LDAPException;



  /**
   * Indicates whether the currently-authenticated user has the specified
   * privilege.  The set of defined privileges may be found in the
   * privilege-list.html and privilege-list.csv files in the server docs
   * directory.
   *
   * @param  privilegeName  The name of the privilege for which to make the
   *                        determination.
   * @param  operation      The operation currently being processed.  It may be
   *                        {@code null} if no operation is available.
   *
   * @return  {@code true} if the currently-authenticated user has the specified
   *          privilege, or {@code false} if not (or if the client is not
   *          authenticated).
   *
   * @throws  LDAPException  If the specified privilege is not defined in the
   *                         server, or if a problem is encountered while trying
   *                         to make the determination.
   */
  boolean hasPrivilege(final String privilegeName,
                       final OperationContext operation)
          throws LDAPException;



  /**
   * Retrieves the names of the privileges held by the currently-authenticated
   * user.  The set of defined privileges may be found in the
   * privilege-list.html and privilege-list.csv files in the server docs
   * directory.
   *
   * @return  The names of the privileges held by the currently-authenticated
   *          user, or an empty set if the authenticated user does not have any
   *          privileges (or if the client is not authenticated).
   */
  Set<String> getPrivilegeNames();



  /**
   * Attempts to send an unsolicited notification to the client with the
   * provided information.
   *
   * @param  oid         The OID for the unsolicited notification.  It must not
   *                     be {@code null}.
   * @param  resultCode  The result code to use for the unsolicited
   *                     notification.  It must not be {@code null}.
   * @param  message     A message to include in the unsolicited notification.
   *                     It may be {@code null} if no message is needed.
   */
  void sendUnsolicitedNotification(final String oid,
                                   final ResultCode resultCode,
                                   final String message);



  /**
   * Terminates the connection to the client and interrupts any operations that
   * may be in progress on that connection.
   *
   * @param  reason        A general reason that the connection was closed.
   * @param  notifyClient  Indicates whether to attempt to send a notice of
   *                       disconnection to the client.
   * @param  message       A message with information about the reason for the
   *                       disconnect.  It may be {@code null} if none is
   *                       available.  It is generally recommended that a
   *                       message be provided even if the client should not be
   *                       notified, since the message may be used in other
   *                       ways (e.g., in log messages).
   */
  void disconnect(final DisconnectReason reason, final boolean notifyClient,
                  final String message);



  /**
   * Retrieves information about the server with which the client connection is
   * associated.
   *
   * @return  Information about the server with which the client connection is
   *          associated.
   */
  ServerContext getServerContext();



  /**
   * Retrieves an opaque object with information about the state of an active
   * multi-stage SASL bind.  The core server will not make any attempt to
   * interpret this object, but it is expected that any SASL mechanism handler
   * which makes use of SASL state information will know how to interact with
   * this object.
   *
   * @return  An opaque object with information about the state of an active
   *          multi-stage SASL bind, or {@code null} if no state information is
   *          available (e.g., because no multi-stage SASL bind is in progress,
   *          or because no state information is needed for the active bind
   *          operation).
   */
  Object getSASLAuthStateInfo();



  /**
   * Sets state information for an active multi-stage SASL bind.  It is
   * recommended that if any SASL state information is set in the connection,
   * then that state should be cleared when it is no longer required (e.g.,
   * after the bind has completed or failed).
   *
   * @param  saslAuthStateInfo  An opaque object that may hold information about
   *                            the state of an active multi-stage SASL bind.
   *                            It may be {@code null} to clear any existing
   *                            SASL authentication state.  The core sever will
   *                            not make any attempt to interpret this object,
   *                            but it is expected that any SASL mechanism
   *                            handler which makes use of SASL state
   *                            information will know how to interact with this
   *                            object.
   */
  void setSASLAuthStateInfo(final Object saslAuthStateInfo);



  /**
   * Retrieves a named object that has been associated with this client
   * connection.
   *
   * @param  name  The name of the attachment to retrieve.  It will be treated
   *               in a case-sensitive manner.  Note that attachment names must
   *               be carefully crafted to avoid inadvertent conflicts between
   *               extensions or the core server itself.  It is strongly
   *               recommended that attachment names be made unique (e.g.,
   *               by prefixing them with the fully-qualified class name of the
   *               extension with which they are associated) so that attachments
   *               used by one extension do not inadvertently interfere with
   *               those which may be used in another extension or elsewhere in
   *               the server.
   *
   * @return  The object that has been associated with this client connection
   *          using the given name, or {@code null} if there is no such
   *          attachment.
   */
  Object getAttachment(final String name);



  /**
   * Attaches an object to this client connection.
   *
   * @param  name   The name of the attachment to retrieve.  It will be treated
   *                in a case-sensitive manner.  Note that attachment names must
   *                be carefully crafted to avoid inadvertent conflicts between
   *                extensions or the core server itself.  It is strongly
   *                recommended that attachment names be made unique (e.g.,
   *                by prefixing them with the fully-qualified class name of the
   *                extension with which they are associated) so that
   *                attachments used by one extension do not inadvertently
   *                interfere with those which may be used in another extension
   *                or elsewhere in the server.
   * @param  value  The attachment to set.  It may be {@code null} if an
   *                existing attachment with the given name should be removed.
   *
   * @return  The attachment value held before the new value was assigned, or
   *          {@code null} if the attachment did not previously have a value.
   */
  Object setAttachment(final String name, final Object value);



  /**
   * Retrieves a string representation of the client connection.
   *
   * @return  A string representation of the client connection.
   */
  String toString();
}
