/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;



import java.util.Set;

import com.unboundid.directory.sdk.common.operation.GenericResult;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be used to obtain
 * information about an operation for which processing has been completed.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_NOT_THREADSAFE)
public interface CompletedOperationContext
       extends OperationContext
{
  /**
   * Retrieves the time that the operation was added to the work queue, if
   * available.    The value returned will be an offset in milliseconds since
   * 12:00 a.m. on January 1, 1970.
   *
   * @return  The time that the operation was added to the work queue, or -1 if
   *          the enqueue time is not available.
   */
  long getEnqueueTime();



  /**
   * Retrieves the time that the operation was removed from the work queue, if
   * available.    The value returned will be an offset in milliseconds since
   * 12:00 a.m. on January 1, 1970.
   *
   * @return  The time that the operation was removed form the work queue, or -1
   *          if the dequeue time is not available.
   */
  long getDequeueTime();



  /**
   * Retrieves the length of time in milliseconds that the operation was held in
   * the work queue before being picked up for processing by a worker thread.
   *
   * @return  The length of time in milliseconds that the operation was held in
   *          the work queue, or -1 if the queue wait time is not available.
   */
  long getQueueWaitTimeMillis();



  /**
   * Retrieves the time that the worker thread began processing the operation.
   * The value returned will be an offset in milliseconds since 12:00 a.m. on
   * January 1, 1970.
   *
   * @return  The time that the worker thread began processing the operation.
   */
  long getProcessingStartTime();



  /**
   * Retrieves the time that the worker thread completed processing for the
   * operation.  The value returned will be an offset in milliseconds since
   * 12:00 a.m. on January 1, 1970.
   *
   * @return  The time that the worker thread completed processing for the
   *          operation.
   */
  long getProcessingEndTime();



  /**
   * Retrieves the length of time in milliseconds that the worker thread spent
   * processing the operation.
   *
   * @return  The length of time in milliseconds that the worker thread spent
   *          processing the operation.
   */
  long getProcessingTimeMillis();



  /**
   * Retrieves the length of time in nanoseconds that the worker thread spent
   * processing the operation.
   *
   * @return  The length of time in nanoseconds that the worker thread spent
   *          processing the operation.
   */
  long getProcessingTimeNanos();



  /**
   * Retrieves information about the result of the operation processing.
   *
   * @return  Information about the result of the operation processing.
   */
  GenericResult getResult();



  /**
   * Retrieves the names of the privileges used in the course of processing the
   * operation.  The set of defined privileges may be found in the
   * privilege-list.html and privilege-list.csv files in the server docs
   * directory.
   *
   * @param  includePreAuthorizationPrivileges  Indicates whether the returned
   *                                            set of privileges should include
   *                                            any privileges used before the
   *                                            assignment of an alternate
   *                                            authorization identity.
   *
   * @return  The names of the privileges used in the course of processing the
   *          operation, or an empty set if no privileges were used.
   */
  Set<String> getUsedPrivilegeNames(
                   final boolean includePreAuthorizationPrivileges);



  /**
   * Retrieves the names of any privileges used before the assignment of an
   * alternate authorization identity.  If no alternate authorization identity
   * was used, then this set will be empty.  If an alternate authorization
   * identity was used, then it should include at least the proxied-auth
   * privilege, but may include other privileges that had been used before the
   * assignment of that alternate authorization identity.
   * <BR><BR>
   * The set of defined privileges may be found in the privilege-list.html and
   * privilege-list.csv files in the server docs directory.
   *
   * @return  The names of any privileges used before the assignment of an
   *          alternate authorization identity, or an set if no alternate
   *          authorization identity was used.
   */
  Set<String> getPreAuthorizationUsedPrivilegeNames();



  /**
   * Retrieves the names of any privileges that were required during the course
   * of processing the operation that the requester did not have.  The set of
   * defined privileges may be found in the privilege-list.html and
   * privilege-list.csv files in the server docs directory.
   *
   * @return  The names of any privileges that were required during the course
   *          of processing the operation that the requester did not have, or an
   *          empty set if the requester was not missing any required
   *          privileges.
   */
  Set<String> getMissingPrivilegeNames();
}
