/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;



/**
 * This enum defines a set of values which may be used to represent the result
 * of a Boolean evaluation in which the result may be undefined.
 */
public enum ConditionResult
{
  /**
   * A condition result value which represents a Boolean result of "true".
   */
  TRUE("true"),



  /**
   * A condition result value which represents a Boolean result of "false".
   */
  FALSE("false"),



  /**
   * A condition result value which represents an undefined result.
   */
  UNDEFINED("undefined");



  // The string representation of this condition result.
  private final String name;



  /**
   * Creates a new condition result value with the provided name.
   *
   * @param  name  The name for this condition result value.
   */
  ConditionResult(final String name)
  {
    this.name = name;
  }



  /**
   * Retrieves the condition result value which is the inverse of this value.
   * The inverse of {@code TRUE} is {@code FALSE}, and vice-versa.  The inverse
   * of {@code UNDEFINED} is {@code UNDEFINED}.
   *
   * @return  The condition result value which is the inverse of this value.
   */
  public ConditionResult invert()
  {
    switch (this)
    {
      case TRUE:
        return FALSE;
      case FALSE:
        return TRUE;
      case UNDEFINED:
      default:
        return UNDEFINED;
    }
  }



  /**
   * Retrieves the condition result value which is the inverse of the provided
   * result.  The inverse of {@code TRUE} is {@code FALSE}, and vice-versa.  The
   * inverse of {@code UNDEFINED} is {@code UNDEFINED}.
   *
   * @param  r  The condition result value for which to retrieve the inverse
   *            value.  It must not be {@code null}.
   *
   * @return  The condition result value which is the inverse of the provided
   *          result.
   */
  public static ConditionResult invert(final ConditionResult r)
  {
    return r.invert();
  }



  /**
   * Retrieves a string representation of this condition result value.
   *
   * @return  A string representation of this condition result value.
   */
  @Override()
  public String toString()
  {
    return name;
  }
}
