/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;



import java.util.Date;
import java.util.Set;

import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be used to interact with an
 * encryption settings definition.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface EncryptionSettingsDefinition
{
  /**
   * Retrieves a string representation of the primary identifier for this
   * encryption settings definition.
   *
   * @return  A string representation of the primary identifier for this
   *          encryption settings definition.
   */
  String getPrimaryIDString();



  /**
   * Retrieves a set of all identifiers that may be used to reference this
   * encryption settings definition, including the primary identifier and any
   * legacy identifiers that may have been used for this definition over time.
   *
   * @return  A set of all identifiers that may be used to reference this
   *          encryption settings definition.
   */
  Set<String> getAllIDStrings();



  /**
   * Retrieves the cipher transformation used for this encryption settings
   * definition.
   *
   * @return  The cipher transformation used for this encryption settings
   *          definition.
   */
  String getCipherTransformation();



  /**
   * Retrieves the length of the GCM authentication tag in bits, if applicable.
   *
   * @return  The length of the GCM authentication tag in bits, or -1 if it does
   *          not apply to this encryption settings definition.
   */
  int getGCMTagLengthBits();



  /**
   * Retrieves the name of the key factory algorithm used to generate the key
   * for this encryption settings definition.
   *
   * @return  The name of the key factory algorithm used to generate the key for
   *          this encryption settings definition.
   */
  String getKeyFactoryAlgorithm();



  /**
   * Retrieves the iteration count provided to the key factory algorithm in the
   * course of generating the encryption key.
   *
   * @return  The iteration count provided to the key factory algorithm in the
   *          course of generating the encryption key.
   */
  int getKeyIterationCount();



  /**
   * Retrieves the size in bytes of the salt provided to the key factory
   * algorithm in the course of generating the encryption key.
   *
   * @return  The size in bytes of the salt provided to the key factory
   *          algorithm in the course of generating the encryption key.
   */
  int getKeySaltLengthBytes();



  /**
   * Retrieves the length in bytes of the initialization vector that will be
   * used with this encryption settings definition.
   *
   * @return  The length in bits of the initialization vector that will be used
   *          with this encryption settings definition.
   */
  int getInitializationVectorLengthBytes();



  /**
   * Retrieves the length in bits of the key for this encryption settings
   * object.
   *
   * @return  The length in bits of the key for this encryption settings object.
   */
  int getKeyLengthBits();



  /**
   * Retrieves the time that this encryption settings definition was created,
   * if available.
   *
   * @return  The time that this encryption settings definition was created, or
   *          {@code null} if that is not available (because the definition was
   *          created before the create timestamp was recorded).
   */
  Date getCreateTime();



  /**
   * Retrieves a description for this encryption settings definition, if
   * available.
   *
   * @return  A description for this encryption settings definition, or
   *          {@code null} if no description is available.
   */
  String getDescription();



  /**
   * Retrieves a string representation of this encryption settings definition.
   *
   * @return  A string representation of this encryption settings definition.
   */
  @Override()
  String toString();
}
