/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;



import java.util.List;
import java.util.Set;

import com.unboundid.directory.sdk.common.schema.AttributeType;
import com.unboundid.ldap.sdk.Attribute;
import com.unboundid.ldap.sdk.DN;
import com.unboundid.ldap.sdk.Filter;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ReadOnlyEntry;
import com.unboundid.ldap.sdk.SearchScope;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods which may be used to interact with an
 * entry.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_NOT_THREADSAFE)
public interface Entry
{
  /**
   * Retrieves the DN of the entry as a string.
   *
   * @return  The DN of the entry as a string.
   */
  String getDN();



  /**
   * Retrieves a parsed representation of the DN for the entry.
   *
   * @return  A parsed representation of the DN for the entry.
   *
   * @throws  LDAPException  If a problem occurs while trying to parse the DN.
   */
  DN getParsedDN()
     throws LDAPException;



  /**
   * Retrieves a list of the attributes contained in the entry.  It may contain
   * both real and virtual attributes.
   *
   * @return  A list of the attributes contained in the entry.
   */
  List<Attribute> getAttributes();



  /**
   * Retrieves a list of the real (i.e., non-virtual) attributes contained in
   * the entry.
   *
   * @return  A list of the real attributes contained in the entry.
   */
  List<Attribute> getRealAttributes();



  /**
   * Retrieves a list of the virtual attributes contained in the entry.
   *
   * @return  A list of the virtual attributes contained in the entry.
   */
  List<Attribute> getVirtualAttributes();



  /**
   * Indicates whether the entry contains an attribute with the specified type.
   *
   * @param  type  The attribute type for which to make the determination.  It
   *               must not be {@code null}.
   *
   * @return  {@code true} if the entry has an attribute with the specified
   *          name, or {@code false} if not.
   */
  boolean hasAttribute(final AttributeType type);



  /**
   * Indicates whether the entry contains an attribute with the specified name.
   *
   * @param  name  The name or OID of the attribute for which to make the
   *               determination.  It must not be {@code null}.
   *
   * @return  {@code true} if the entry has an attribute with the specified
   *          name, or {@code false} if not.
   */
  boolean hasAttribute(final String name);



  /**
   * Indicates whether the entry contains an attribute with the specified name
   * and set of options.
   *
   * @param  type     The attribute type for which to make the determination.
   *                  It must not be {@code null}.
   * @param  options  The set of attribute options for which to make the
   *                  determination.  It may be {@code null} or empty to
   *                  indicate no options.
   *
   * @return  {@code true} if the entry has an attribute with the specified
   *          name, or {@code false} if not.
   */
  boolean hasAttribute(final AttributeType type, final Set<String> options);



  /**
   * Indicates whether the entry contains an attribute with the specified name
   * and set of options.
   *
   * @param  name     The name or OID of the attribute for which to make the
   *                  determination.  It must not be {@code null}.
   * @param  options  The set of attribute options for which to make the
   *                  determination.  It may be {@code null} or empty to
   *                  indicate no options.
   *
   * @return  {@code true} if the entry has an attribute with the specified
   *          name, or {@code false} if not.
   */
  boolean hasAttribute(final String name, final Set<String> options);



  /**
   * Indicates whether the entry contains an attribute with the specified name
   * and value.
   *
   * @param  type   The attribute type for the attribute for which to make the
   *                determination.  It must not be {@code null}.
   * @param  value  The value for which to make the determination.  It must not
   *                be {@code null}.
   *
   * @return  {@code true} if the entry has an attribute with the specified name
   *          and value, or {@code false} if not.
   */
  boolean hasAttributeValue(final AttributeType type, final String value);



  /**
   * Indicates whether the entry contains an attribute with the specified name
   * and value.
   *
   * @param  name   The name or OID of the attribute for which to make the
   *                determination.  It must not be {@code null}.
   * @param  value  The value for which to make the determination.  It must not
   *                be {@code null}.
   *
   * @return  {@code true} if the entry has an attribute with the specified name
   *          and value, or {@code false} if not.
   */
  boolean hasAttributeValue(final String name, final String value);



  /**
   * Indicates whether the entry contains an attribute with the specified name
   * and value.
   *
   * @param  type   The attribute type for the attribute for which to make the
   *                determination.  It must not be {@code null}.
   * @param  value  The value for which to make the determination.  It must not
   *                be {@code null}.
   *
   * @return  {@code true} if the entry has an attribute with the specified name
   *          and value, or {@code false} if not.
   */
  boolean hasAttributeValue(final AttributeType type, final byte[] value);



  /**
   * Indicates whether the entry contains an attribute with the specified name
   * and value.
   *
   * @param  name   The name or OID of the attribute for which to make the
   *                determination.  It must not be {@code null}.
   * @param  value  The value for which to make the determination.  It must not
   *                be {@code null}.
   *
   * @return  {@code true} if the entry has an attribute with the specified name
   *          and value, or {@code false} if not.
   */
  boolean hasAttributeValue(final String name, final byte[] value);



  /**
   * Retrieves the attribute with the specified name.  This will include all
   * variants of the attribute type with different sets of attribute options.
   *
   * @param  type  The attribute type for the attribute to retrieve.  It must
   *               not be {@code null}.
   *
   * @return  A list of all occurrences of the requested attribute with any set
   *          of options, or {@code null} if the specified attribute is not
   *          present in the entry.
   */
  List<Attribute> getAttribute(final AttributeType type);



  /**
   * Retrieves the attribute with the specified name.  This will include all
   * variants of the attribute type with different sets of attribute options.
   *
   * @param  name  The name or OID of the attribute to retrieve.  It must not be
   *               {@code null}.
   *
   * @return  A list of all occurrences of the requested attribute with any set
   *          of options, or {@code null} if the specified attribute is not
   *          present in the entry.
   */
  List<Attribute> getAttribute(final String name);



  /**
   * Retrieves the attribute with the specified name and exact set of options.
   *
   * @param  type     The attribute type for the attribute to retrieve.  It must
   *                  not be {@code null}.
   * @param  options  The set of attribute options for the attribute to
   *                  retrieve.  It may be {@code null} or empty if there should
   *                  not be any options.
   *
   * @return  The requested attribute, or {@code null} if it is not present in
   *          the entry.
   */
  Attribute getAttribute(final AttributeType type, final Set<String> options);



  /**
   * Retrieves the attribute with the specified name and exact set of options.
   *
   * @param  name     The name or OID of the attribute to retrieve.  It must not
   *                  be {@code null}.
   * @param  options  The set of attribute options for the attribute to
   *                  retrieve.  It may be {@code null} or empty if there should
   *                  not be any options.
   *
   * @return  The requested attribute, or {@code null} if it is not present in
   *          the entry.
   */
  Attribute getAttribute(final String name, final Set<String> options);



  /**
   * Indicates whether this entry is within the range of the provided base DN
   * and scope.
   *
   * @param  baseDN  The base DN for which to make the determination.
   * @param  scope   The scope for which to make the determination.
   *
   * @return  {@code true} if this entry is within the range specified by the
   *          provided base DN and scope, or {@code false} if not.
   *
   * @throws  LDAPException  If the provided string cannot be parsed as a valid
   *                         DN.
   */
  boolean matchesBaseAndScope(final String baseDN, final SearchScope scope)
          throws LDAPException;



  /**
   * Indicates whether this entry matches the provided filter.
   *
   * @param  filter  The filter for which to make the determination.  It must
   *                 not be {@code null}.
   *
   * @return  {@code true} if this entry matches the provided filter, or
   *          {@code false} if not.
   *
   * @throws  LDAPException  If a problem occurs while making the determination,
   *                         or if the provided string cannot be parsed as a
   *                         valid filter.
   */
  boolean matchesFilter(final String filter)
          throws LDAPException;



  /**
   * Indicates whether this entry matches the provided filter.
   *
   * @param  filter  The filter for which to make the determination.  It must
   *                 not be {@code null}.
   *
   * @return  {@code true} if this entry matches the provided filter, or
   *          {@code false} if not.
   *
   * @throws  LDAPException  If a problem occurs while making the determination,
   *                         or if the provided string cannot be parsed as a
   *                         valid filter.
   */
  boolean matchesFilter(final Filter filter)
          throws LDAPException;



  /**
   * Converts this server entry to its corresponding LDAP SDK representation.
   *
   * @return  An LDAP SDK representation of this server entry.
   */
  ReadOnlyEntry toLDAPSDKEntry();
}
