/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2011-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;


import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URL;
import java.security.CodeSource;
import java.util.jar.JarFile;

/**
 * This class represents the physical state of a ${SHORT_VENDOR_NAME} server
 * extension bundle.  All the operations are dependent upon the root directory
 * that is specified in the constructor.
 */
public class ExtensionBundle
{
  /**
   * The name of the jar file manifest attribute that provides the extension
   * bundle name.
   */
  public static final String MANIFEST_ATTR_TITLE =
       "Implementation-Title";



  /**
   * The name of the jar file manifest attribute that provides the extension
   * bundle version.
   */
  public static final String MANIFEST_ATTR_VERSION =
       "Implementation-Version";



  /**
   * The name of the jar file manifest attribute that provides the extension
   * bundle vendor.
   */
  public static final String MANIFEST_ATTR_VENDOR =
       "Implementation-Vendor";



  /**
   * The name of the jar file manifest attribute that provides the extension
   * bundle vendor ID.
   */
  public static final String MANIFEST_ATTR_VENDOR_ID =
       "Implementation-Vendor-Id";



  /**
   * The name of the jar file manifest attribute that provides the extension
   * bundle URL.
   */
  public static final String MANIFEST_ATTR_URL =
       "Implementation-URL";



  /**
   * The name of the jar file manifest attribute that provides the extension
   * bundle support contact.
   */
  public static final String MANIFEST_ATTR_SUPPORT_CONTACT =
       "Extension-Support-Contact";



  /**
   * The name of the jar file manifest attribute that provides the server SDK
   * version number.
   */
  public static final String MANIFEST_ATTR_SERVER_SDK_VERSION =
       "UnboundID-Server-SDK-Version";

  /**
   * The relative path where the config files are.
   */
  public static final String CONFIG_PATH_RELATIVE = "config";

  /**
   * The relative path where all the libraries (jar files) are.
   */
  public static final String LIBRARIES_PATH_RELATIVE = "lib";

  /**
   * The relative path where all the documentation files are.
   */
  public static final String DOCUMENTATION_PATH_RELATIVE = "docs";

  /**
   * The relative path where the config files are.
   */
  public static final String HISTORY_PATH_RELATIVE = "history";

  /**
   * Path to the config/update directory where update base files
   * are stored.
   */
  public static final String UPDATE_PATH = "update";

  /**
   * File written to history/[timestamp]-[version] to document whatever
   * modifications were made to the file system during an update.
   */
  public static final String UPDATE_LOG_NAME = "update.log";

  // A FileFilter for JAR files.
  private static final FileFilter JAR_FILTER = new FileFilter()
  {

    /**
     * Must be a Jar file.
     */
    public boolean accept(final File pathname)
    {
      if (!pathname.isFile())
      {
        return false;
      }

      String name = pathname.getName();
      return name.endsWith(".jar");
    }

  };

  private final File extensionBundleDir;
  private final File extensionJarFile;
  private final java.util.jar.Manifest manifest;
  private final String vendorId;
  private final String title;
  private final String version;
  private final String bundleId;
  private final String sdkVersion;

  /**
   * Creates a new instance from a root directory specified as a File.
   *
   * @param extensionBundleDir root directory of this extension bundle.
   * @throws IOException if a error occurs while opening the bundle.
   * @throws IllegalArgumentException if a error occurs while loading the
   *         extension JAR.
   */
  public ExtensionBundle(final File extensionBundleDir) throws IOException,
      IllegalArgumentException {
    File jarFile = null;
    java.util.jar.Manifest jarManifest = null;
    String extensionVendorId = null;
    String extensionTitle = null;
    String extensionVersion = null;
    String extensionSdkVersion = null;
    boolean isValid = false;

    final File[] files = extensionBundleDir.listFiles(JAR_FILTER);
    if (files != null)
    {
      for (File file : extensionBundleDir.listFiles(JAR_FILTER)) {
        jarFile = file;
        JarFile jar = new JarFile(file);
        jarManifest = jar.getManifest();
        extensionVendorId = jarManifest.getMainAttributes().getValue(
            MANIFEST_ATTR_VENDOR_ID);
        extensionTitle = jarManifest.getMainAttributes().getValue(
            MANIFEST_ATTR_TITLE);
        extensionVersion = jarManifest.getMainAttributes().getValue(
            MANIFEST_ATTR_VERSION);
        extensionSdkVersion = jarManifest.getMainAttributes().getValue(
            MANIFEST_ATTR_SERVER_SDK_VERSION);
        if (extensionVendorId == null || extensionVendorId.isEmpty()) {
          continue;
        }
        if (extensionTitle == null || extensionTitle.isEmpty()) {
          continue;
        }
        if (extensionVersion == null || extensionVersion.isEmpty()) {
          continue;
        }
        if (extensionSdkVersion == null || extensionSdkVersion.isEmpty()) {
          continue;
        }
        isValid = true;
        break;
      }
    }
    if(!isValid)
    {
      throw new FileNotFoundException("Cannot find a valid extension jar in " +
          "bundle");
    }

    checkCompatibility(extensionSdkVersion);

    this.extensionBundleDir = extensionBundleDir;
    this.extensionJarFile = jarFile;
    this.manifest = jarManifest;
    this.vendorId = extensionVendorId;
    this.title = extensionTitle;
    this.version = extensionVersion;
    this.sdkVersion = extensionSdkVersion;
    this.bundleId = extensionVendorId + "." + extensionTitle;
  }

  /**
   * Retrieves the title of this extension bundle.
   *
   * @return The title of this extension bundle.
   */
  public String getTitle()
  {
    return title;
  }

  /**
   * Retrieves the vendor name of this extension bundle.
   *
   * @return The vendor name of this extension bundle.
   */
  public String getVendor()
  {
    return manifest.getMainAttributes().getValue(MANIFEST_ATTR_VENDOR);
  }

  /**
   * Retrieves the vendor ID of this extension bundle.
   *
   * @return The vendor ID of this extension bundle.
   */
  public String getVendorId()
  {
    return vendorId;
  }

  /**
   * Retrieves the version string of this extension bundle.
   *
   * @return The version string of this extension bundle.
   */
  public String getVersion()
  {
    return version;
  }

  /**
   * Get the server SDK version number the extensions where built against.
   *
   * @return The server SDK version number the extensions where built against.
   */
  public String getServerSDKVersion()
  {
    return sdkVersion;
  }


  /**
   * Get the URL of the extension bundle.
   *
   * @return The URL of the extension bundle.
   */
  public String getUrl()
  {
    return manifest.getMainAttributes().getValue(MANIFEST_ATTR_URL);
  }

  /**
   * Get the support contact for this extension bundle.
   *
   * @return The support contact for this extension bundle.
   */
  public String getSupportContact()
  {
    return manifest.getMainAttributes().getValue(MANIFEST_ATTR_SUPPORT_CONTACT);
  }

  /**
   * Get the ID string that could be used to uniquely identify this bundle.
   *
   * @return The ID string that could be used to uniquely identify this bundle.
   */
  public String getBundleId()
  {
    return bundleId;
  }

  /**
   * Get the root directory of this extension bundle.
   *
   * @return The root directory of this extension bundle.
   */
  public File getExtensionBundleDir()
  {
    return extensionBundleDir;
  }

  /**
   * Get the extension jar file in the bundle.
   *
   * @return The extension jar file in the bundle.
   */
  public File getExtensionJarFile()
  {
    return extensionJarFile;
  }

  /**
   * Get the configuration directory for this extension bundle.
   *
   * @return The configuration directory for this extension bundle.
   */
  public File getConfigDir()
  {
    return new File(extensionBundleDir, CONFIG_PATH_RELATIVE);
  }

  /**
   * Get the update directory for this extension bundle.
   *
   * @return The update directory for this extension bundle.
   */
  public File getConfigUpdateDir()
  {
    return new File(getConfigDir(), UPDATE_PATH);
  }

  /**
   * Get the lib directory for this extension bundle.
   *
   * @return The lib directory for this extension bundle.
   */
  public File getLibrariesDir()
  {
    return new File(extensionBundleDir, LIBRARIES_PATH_RELATIVE);
  }

  /**
   * Get the documentation directory for this extension bundle.
   *
   * @return The documentation directory for this extension bundle.
   */
  public File getDocumentationDir()
  {
    return new File(extensionBundleDir, DOCUMENTATION_PATH_RELATIVE);
  }

  /**
   * Get the history directory for this extension bundle.
   *
   * @return The history directory for this extension bundle.
   */
  public File getHistoryDir()
  {
    return new File(extensionBundleDir, HISTORY_PATH_RELATIVE);
  }

  /**
   * Checks to make sure the SDK version specified in the extension JAR file
   * is compatible with this server instance.
   *
   * @param serverSDKVersionString The server SDK version string.
   * @throws IllegalArgumentException if the version string is invalid or
   * not compatible.
   */
  public static void checkCompatibility(final String serverSDKVersionString)
      throws IllegalArgumentException
  {
    final int jarSDKMajor;
    final int jarSDKMinor;
    final int jarSDKPoint;
    final int jarSDKPatch;
    try
    {
      final String[] versionComps = serverSDKVersionString.split("\\.");
      jarSDKMajor = Integer.parseInt(versionComps[0]);
      jarSDKMinor = Integer.parseInt(versionComps[1]);
      jarSDKPoint = Integer.parseInt(versionComps[2]);
      jarSDKPatch = Integer.parseInt(versionComps[3]);
    }
    catch (final Exception e)
    {
      throw new IllegalArgumentException("Value '" + serverSDKVersionString +
          "' for manifest attribute '" + MANIFEST_ATTR_SERVER_SDK_VERSION +
          "' is malformed", e);
    }

    boolean acceptable = true;
    if (jarSDKMajor > Version.MAJOR_VERSION)
    {
      acceptable = false;
    }
    else if (jarSDKMajor == Version.MAJOR_VERSION)
    {
      if (jarSDKMinor > Version.MINOR_VERSION)
      {
        acceptable = false;
      }
      else if (jarSDKMinor == Version.MINOR_VERSION)
      {
        if (jarSDKPoint > Version.POINT_VERSION)
        {
          acceptable = false;
        }
        else if (jarSDKPoint == Version.POINT_VERSION)
        {
          acceptable = jarSDKPatch <= Version.PATCH_VERSION;
        }
      }
    }

    if (! acceptable)
    {
      final String maxSupportedVersion =
          Version.MAJOR_VERSION + "." + Version.MINOR_VERSION + '.' +
              Version.POINT_VERSION + '.' + Version.PATCH_VERSION;
      throw new IllegalArgumentException("Extensions built using " +
          Version.PRODUCT_NAME + " version " + serverSDKVersionString + ", " +
          "which is newer than the maximum server SDK version of " +
          maxSupportedVersion);
    }
  }

  /**
   * Checks if the given Class instance represents a class
   * packaged in this extension bundle.
   *
   * @param clazz An class loaded into the server
   * @return true if the given class is included in this extension bundle
   */
  public boolean containsClass(final Class clazz)
  {
    try
    {
      URL url = extensionJarFile.toURI().toURL();
      CodeSource codeSource = clazz.getProtectionDomain().getCodeSource();
      if(codeSource != null)
      {
        return codeSource.getLocation().sameFile(url);
      }
    }
    catch(Exception e)
    {
      // return false below
    }
    return false;
  }
}
