/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2017-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;



import com.unboundid.ldap.sdk.DN;
import com.unboundid.ldap.sdk.Filter;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.SearchScope;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be used to interact with a
 * group.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_NOT_THREADSAFE)
public interface Group
{
  /**
   * Retrieves the DN of the entry that defines this group.
   *
   * @return  The DN of the entry that defines this group.
   */
  DN getDN();



  /**
   * Indicates whether the specified user is a member of this group.
   *
   * @param  userDN                The DN for the user for whom to make the
   *                               determination.  It must not be {@code null}.
   * @param  directMembershipOnly  Indicates whether to only consider the user a
   *                               member of the group if that user is directly
   *                               named as a member of the group.  If this
   *                               parameter is {@code true}, then the method
   *                               will return {@code true} only if the user is
   *                               directly listed as a member of the group, and
   *                               will not included dynamic or nested
   *                               membership.  If this parameter is
   *                               {@code false}, then the method will return
   *                               {@code true} if the user is a direct member
   *                               of the group, is a member of a dynamic group
   *                               in which the user matches the membership
   *                               criteria, or if the user is a member of a
   *                               nested group.
   *
   * @return  {@code true} if the specified user is a member of this group, or
   *          {@code false} if not.
   *
   * @throws  LDAPException  If a problem is encountered while attempting to
   *                         make the determination.
   */
  boolean isMember(final String userDN, final boolean directMembershipOnly)
          throws LDAPException;



  /**
   * Indicates whether the specified user is a member of this group.
   *
   * @param  userEntry             The entry for the user for whom to make the
   *                               determination.  It must not be {@code null}.
   * @param  directMembershipOnly  Indicates whether to only consider the user a
   *                               member of the group if that user is directly
   *                               named as a member of the group.  If this
   *                               parameter is {@code true}, then the method
   *                               will return {@code true} only if the user is
   *                               directly listed as a member of the group, and
   *                               will not included dynamic or nested
   *                               membership.  If this parameter is
   *                               {@code false}, then the method will return
   *                               {@code true} if the user is a direct member
   *                               of the group, is a member of a dynamic group
   *                               in which the user matches the membership
   *                               criteria, or if the user is a member of a
   *                               nested group.
   *
   * @return  {@code true} if the specified user is a member of this group, or
   *          {@code false} if not.
   *
   * @throws  LDAPException  If a problem is encountered while attempting to
   *                         make the determination.
   */
  boolean isMember(final Entry userEntry, final boolean directMembershipOnly)
          throws LDAPException;



  /**
   * Indicates whether the
   * {@link #getMemberIterator(String,SearchScope,Filter,boolean)} method can be
   * used to efficiently obtain information about all members of this group.
   *
   * @return  {@code true} if the {@code getMembers} method should be able to
   *          efficiently retrieve a list of all members of this group, or
   *          {@code false} if the call to {@code getMembers} may be
   *          inefficient.
   */
  boolean canRetrieveMemberListEfficiently();



  /**
   * Retrieves an object that may be used to iterate through the members of this
   * group.  It may be used to iterate through all members, or only members that
   * match a specified set of criteria.
   *
   * @param  baseDN                A base DN that may be used to restrict the
   *                               set of members that are made available
   *                               through the iterator.  If this is
   *                               non-{@code null}, then the iterator will only
   *                               iterate across entries at or below this base
   *                               DN that are also within the specified scope.
   * @param  scope                 A search scope that may be used to restrict
   *                               the set of members that are made available
   *                               through the iterator.  If the base DN is
   *                               non-{@code null}, then the scope must also be
   *                               non-{@code null}, and the iterator will only
   *                               iterate across entries at or below the base
   *                               DN and within this scope.
   * @param  filter                A filter that may be used to restrict the
   *                               set of members that are made available
   *                               through the iterator.  If this is
   *                               non-{@code null}, then the iterator will only
   *                               iterate across members that match this
   *                               filter.
   * @param  includeNestedMembers  Indicates whether to include members that are
   *                               not direct members of this group, but that
   *                               are nested members because they are members
   *                               of a group that is directly or indirectly a
   *                               member of this group.
   *
   * @return  A {@code GroupMemberIterator} object that may be used to iterate
   *          across the members of this group.
   *
   * @throws  LDAPException  If a problem is encountered while creating the
   *                         requested group member iterator.
   */
  GroupMemberIterator getMemberIterator(final String baseDN,
                                        final SearchScope scope,
                                        final Filter filter,
                                        final boolean includeNestedMembers)
                      throws LDAPException;
}
