/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2017-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;



import java.io.Closeable;

import com.unboundid.ldap.sdk.DN;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be used to iterate through
 * the members of a group.  The {@link #close()} method should be called if the
 * caller wants to stop iterating before reaching the end of the set of members.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_NOT_THREADSAFE)
public interface GroupMemberIterator
       extends Closeable
{
  /**
   * Retrieves the DN of the entry for the next group member.  This method will
   * not necessarily ensure that the DN refers to an entry that exists.
   *
   * @return  The DN of the next group member, or {@code null} if there are no
   *          more members.
   *
   * @throws  GroupMemberIteratorException  If a problem is encountered while
   *               attempting to retrieve the DN for the next member.  The
   *               {@link GroupMemberIteratorException#mayContinueIterating()}
   *               method should be used to determine whether the problem is
   *               with the next member and it is safe to continue using this
   *               iterator, or whether the iterator itself is no longer valid.
   */
  DN nextMemberDN()
     throws GroupMemberIteratorException;



  /**
   * Retrieves the entry for the next group member.
   *
   * @return  The entry for the next group member, or {@code null} if there are
   *          no more members.
   *
   * @throws  GroupMemberIteratorException  If a problem is encountered while
   *               attempting to retrieve the entry for the next member.  The
   *               {@link GroupMemberIteratorException#mayContinueIterating()}
   *               method should be used to determine whether the problem is
   *               with the next member and it is safe to continue using this
   *               iterator, or whether the iterator itself is no longer valid.
   */
  Entry nextMemberEntry()
        throws GroupMemberIteratorException;



  /**
   * Closes this iterator and indicates that the server may clean up any
   * resources associated with it.  This method must be used if the caller
   * wishes to stop iterating through the list of members before the end has
   * been reached.  It may also be called after reaching the end of the member
   * list with no ill effects, but that is not necessary.  It may also be called
   * multiple times for the same iterator with no ill effects.
   */
  @Override()
  void close();
}
