/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;



import com.unboundid.asn1.ASN1OctetString;
import com.unboundid.ldap.sdk.BindResult;
import com.unboundid.ldap.sdk.ExtendedRequest;
import com.unboundid.ldap.sdk.ExtendedResult;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.LDAPInterface;
import com.unboundid.ldap.sdk.SimpleBindRequest;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be used to perform internal
 * operations within the server.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_NOT_THREADSAFE)
public interface InternalConnection
       extends LDAPInterface
{
  /**
   * Processes a simple bind request with the provided DN and password.
   * <BR><BR>
   * The LDAP protocol specification forbids clients from attempting to perform
   * a bind on a connection in which one or more other operations are already in
   * progress.  If a bind is attempted while any operations are in progress,
   * then the directory server may or may not abort processing for those
   * operations, depending on the type of operation and how far along the
   * server has already gotten while processing that operation (unless the bind
   * request is one that will not cause the server to attempt to change the
   * identity of this connection, for example by including the retain identity
   * request control in the bind request).  It is recommended that all active
   * operations be abandoned, canceled, or allowed to complete before attempting
   * to perform a bind on an active connection.
   *
   * @param  bindDN    The bind DN for the bind operation.
   * @param  password  The password for the simple bind operation.
   *
   * @return  The result of processing the bind operation.
   *
   * @throws  LDAPException  If the server rejects the bind request, or if a
   *                         problem occurs while sending the request or reading
   *                         the response.
   */
  BindResult bind(final String bindDN, final String password)
             throws LDAPException;



  /**
   * Processes the provided bind request.
   * <BR><BR>
   * The LDAP protocol specification forbids clients from attempting to perform
   * a bind on a connection in which one or more other operations are already in
   * progress.  If a bind is attempted while any operations are in progress,
   * then the directory server may or may not abort processing for those
   * operations, depending on the type of operation and how far along the
   * server has already gotten while processing that operation (unless the bind
   * request is one that will not cause the server to attempt to change the
   * identity of this connection, for example by including the retain identity
   * request control in the bind request).  It is recommended that all active
   * operations be abandoned, canceled, or allowed to complete before attempting
   * to perform a bind on an active connection.
   *
   * @param  bindRequest  The bind request to be processed.  It must not be
   *                      {@code null}.
   *
   * @return  The result of processing the bind operation.
   *
   * @throws  LDAPException  If the server rejects the bind request, or if a
   *                         problem occurs while sending the request or reading
   *                         the response.
   */
  BindResult bind(final SimpleBindRequest bindRequest)
             throws LDAPException;



  /**
   * Processes an extended operation with the provided request OID.  Note that
   * when processing an extended operation, the server will never throw an
   * exception for a failed response -- only when there is a problem sending the
   * request or reading the response.  It is the responsibility of the caller to
   * determine whether the operation was successful.
   *
   * @param  requestOID  The OID for the extended request to process.  It must
   *                     not be {@code null}.
   *
   * @return  The extended result object that provides information about the
   *          result of the request processing.  It may or may not indicate that
   *          the operation was successful.  Note that the extended result will
   *          always be a generic result, even if the LDAP SDK normally returns
   *          a specific subclass.
   *
   * @throws  LDAPException  If a problem occurs while sending the request or
   *                         reading the response.
   */
  ExtendedResult processExtendedOperation(final String requestOID)
                 throws LDAPException;



  /**
   * Processes an extended operation with the provided request OID and value.
   * Note that when processing an extended operation, the server will never
   * throw an exception for a failed response -- only when there is a problem
   * sending the request or reading the response.  It is the responsibility of
   * the caller to determine whether the operation was successful.
   *
   * @param  requestOID    The OID for the extended request to process.  It must
   *                       not be {@code null}.
   * @param  requestValue  The encoded value for the extended request to
   *                       process.  It may be {@code null} if there does not
   *                       need to be a value for the requested operation.
   *
   * @return  The extended result object that provides information about the
   *          result of the request processing.  It may or may not indicate that
   *          the operation was successful.  Note that the extended result will
   *          always be a generic result, even if the LDAP SDK normally returns
   *          a specific subclass.
   *
   * @throws  LDAPException  If a problem occurs while sending the request or
   *                         reading the response.
   */
  ExtendedResult processExtendedOperation(final String requestOID,
                                          final ASN1OctetString requestValue)
                 throws LDAPException;



  /**
   * Processes the provided extended request.  Note that when processing an
   * extended operation, the server will never throw an exception for a failed
   * response -- only when there is a problem sending the request or reading the
   * response.  It is the responsibility of the caller to determine whether the
   * operation was successful.
   *
   * @param  extendedRequest  The extended request to be processed.  It must not
   *                          be {@code null}.
   *
   * @return  The extended result object that provides information about the
   *          result of the request processing.  It may or may not indicate that
   *          the operation was successful.  Note that the extended result will
   *          always be a generic result, even if the LDAP SDK normally returns
   *          a specific subclass.
   *
   * @throws  LDAPException  If a problem occurs while sending the request or
   *                         reading the response.
   */
  ExtendedResult processExtendedOperation(final ExtendedRequest extendedRequest)
                 throws LDAPException;
}
