/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;



import com.unboundid.directory.sdk.common.api.OperationCompletedListener;
import com.unboundid.directory.sdk.common.operation.Request;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;

import java.util.List;
import java.util.Map;


/**
 * This interface defines a set of methods that may be used to obtain
 * information about an operation being processed by the server.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_NOT_THREADSAFE)
public interface OperationContext
{
  /**
   * Retrieves the identifier that has been assigned to the associated client
   * connection.
   *
   * @return  The identifier that has been assigned to the associated client
   *          connection.
   */
  long getConnectionID();



  /**
   * Retrieves the identifier that has been assigned to this operation for the
   * associated client connection.
   *
   * @return  The identifier that has been assigned to this operation for the
   *          associated client connection.
   */
  long getOperationID();




  /**
   * Retrieves the message ID for this operation from the client request.
   *
   * @return  The message ID for this operation from the client request.
   */
  int getMessageID();



  /**
   * Retrieves the operation type for this operation.
   *
   * @return  The operation type for this operation.
   */
  OperationType getOperationType();



  /**
   * Retrieves the request for the associated operation.  The caller should not
   * make any attempt to alter the request that is returned.
   *
   * @return  The request for the associated operation.
   */
  Request getRequest();



  /**
   * Retrieves information about the server in which this operation is being
   * processed.
   *
   * @return  Information about the server in which this operation is being
   *          processed.
   */
  ServerContext getServerContext();



  /**
   * Indicates whether this operation was initiated within the server rather
   * than by an external client.
   *
   * @return  {@code true} if this operation was initiated within the server, or
   *          {@code false} if it was requested by an external client.
   */
  boolean isInternalOperation();



  /**
   * Indicates whether this is an administrative operation. An operation is
   * considered administrative if the
   * {@code StartAdministrativeSessionExtendedRequest} extended request was
   * used on the connection, or the
   * {@code AdministrativeOperationRequestControl} request control was used on
   * the operation.
   *
   * @return  {@code true} if this operation is an administrative operation.
   */
  boolean isAdministrativeOperation();



  /**
   * Indicates whether this operation was initiated via replication rather than
   * within the local server.
   *
   * @return  {@code true} if this operation was initiated via replication, or
   *          {@code false} if it was initiated locally within this server.
   */
  boolean isReplicationOperation();



  /**
   * Retrieves the DN of the user as whom the request is authorized.  Note that
   * this may change over the course of processing the operation (e.g., if the
   * request includes a proxied authorization or intermediate client control).
   *
   * @return  The DN of the user as whom the request is authorized.  It may be
   *          an empty string if the operation is to be authorized as an
   *          unauthenticated user.
   */
  String getAuthorizationDN();



  /**
   * Retrieves the object attached to the associated operation with the given
   * name.
   *
   * @param  name  The case-sensitive name of the attachment to retrieve.  It
   *               must not be {@code null}.
   *
   * @return  The requested attachment, or {@code null} if the operation does
   *          not have an attachment with the provided name.
   */
  Object getAttachment(final String name);



  /**
   * Sets an attachment for the associated operation.
   *
   * @param  name   The name of the attachment to set.  It must not be
   *                {@code null}.
   * @param  value  The value to set for the attachment.  It may be {@code null}
   *                if any existing attachment with the specified name should be
   *                removed.
   */
  void setAttachment(final String name, final Object value);



  /**
   * Retrieves information about the client that requested this operation.
   *
   * @return  Information about the client that requested this operation.
   */
  ClientContext getClientContext();



  /**
   * Indicates whether this operation was requested over a secure connection.
   * If the request included the intermediate client control, then its value
   * will also be used in determining whether communication with downstream
   * clients is also secure.
   *
   * @return  {@code true} if this operation was requested over a secure
   *          connection, or {@code false} if not.
   */
  boolean isSecure();



  /**
   * Retrieves an internal connection that is authenticated as the same user
   * that requested this operation and may be subject to access control.
   * <BR><BR>
   * Note that the returned connection will use the client connection policy
   * defined as the default policy for internal operations.  If you wish to use
   * the client connection policy associated with the connection on which this
   * operation was requested, use the
   * {@link #getInternalUserConnection(boolean)} method.
   *
   * @return  An internal connection that is authenticated as the same user that
   *          requested this operation.
   *
   * @throws  LDAPException  If a problem occurs while attempting to obtain or
   *                         authenticate the connection.
   */
  InternalConnection getInternalUserConnection()
       throws LDAPException;



  /**
   * Retrieves an internal connection that is authenticated as the same user
   * that requested this operation and may be subject to access control.  It may
   * optionally use the client connection policy from the associated client
   * connection.
   *
   * @param  usePolicyFromConnection  If {@code true}, the internal connection
   *                                  will use the same client connection policy
   *                                  as the associated client connection.  If
   *                                  {@code false}, the internal connection
   *                                  will use the server's default client
   *                                  connection policy for internal
   *                                  connections.
   *
   * @return  An internal connection that is authenticated as the same user that
   *          requested this operation.
   *
   * @throws  LDAPException  If a problem occurs while attempting to obtain or
   *                         authenticate the connection.
   */
  InternalConnection getInternalUserConnection(
                          final boolean usePolicyFromConnection)
       throws LDAPException;



  /**
   * Retrieves an internal connection that is authenticated as a root user
   * that is not subject to access control.
   * <BR><BR>
   * Note that the returned connection will use the client connection policy
   * defined as the default policy for internal operations.  If you wish to use
   * the client connection policy associated with the connection on which this
   * operation was requested, use the
   * {@link #getInternalRootConnection(boolean)} method.
   *
   * @return  An internal connection that is authenticated as a root user.
   */
  InternalConnection getInternalRootConnection();



  /**
   * Retrieves an internal connection that is authenticated as a root user
   * that is not subject to access control.  It may optionally use the client
   * connection policy from the associated client connection.
   *
   * @param  usePolicyFromConnection  If {@code true}, the internal connection
   *                                  will use the same client connection policy
   *                                  as the associated client connection.  If
   *                                  {@code false}, the internal connection
   *                                  will use the server's default client
   *                                  connection policy for internal
   *                                  connections.
   *
   * @return  An internal connection that is authenticated as a root user.
   */
  InternalConnection getInternalRootConnection(
       final boolean usePolicyFromConnection);



  /**
   * Retrieves a string representation of this operation.
   *
   * @return  A string representation of this operation.
   */
  String toString();



  /**
   * Specifies an additional log element for this operation, which
   * should be written to the log.
   *  @param  key    The key of the log element being added
   *  @param  value  The value of the log element being added
   */
  void addCustomLogElement(String key, Object value);


  /**
   * Returns any additional log elements for this operation, which
   * should be written to the log.
   *
   * @return map of additional log elements to a list of values.
   */
  Map<String, List<String>> getCustomLogElements();



  /**
   * Indicates that the provided message should be included in the access log
   * message for the operation but should not be returned to the client.
   *
   * @param  message  The message to include in the log message.  It must not
   *                  be {@code null}.
   */
  void appendAdditionalLogMessage(final String message);



  /**
   * Registers the provided listener to be notified when the server completes
   * an operation.
   *
   * @param  listener  The server operation completed listener to be registered.
   *                   It must not be {@code null}.
   *
   * @return  An object representing the operation completed listener that has
   *          been registered with the server.
   */
  RegisteredOperationCompletedListener registerOperationCompletedListener(
      final OperationCompletedListener listener);



  /**
   * Deregisters the provided server operation completed listener with the
   * server.  This will have no effect if the provided listener is not
   * registered.
   *
   * @param  listener  The server operation completed listener to be
   *                   deregistered.  It must not be {@code null}.
   */
  void deregisterOperationCompletedListener(
      final RegisteredOperationCompletedListener listener);
}
