/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2011-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;



import java.io.Serializable;
import java.util.List;

import com.unboundid.util.NotMutable;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This class defines a structure which may be used to provide information about
 * the result of the processing performed by a manage extension plugin.
 */
@NotMutable()
@ThreadSafety(level=ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public final class PostManageExtensionPluginResult
       implements Serializable
{
  /**
   * A predefined result instance that indicates all processing completed
   * successfully.
   */
  public static final PostManageExtensionPluginResult SUCCESS =
       new PostManageExtensionPluginResult(true, null);


  /**
   * The serial version UID for this serializable class.
   */
  private static final long serialVersionUID = 5891391890895925978L;

  // Indicates whether the manage-extension tool should restart the server
  // after the installation if it was previous started.
  private final boolean restartServer;

  private final List<String> messages;



  /**
   * Creates a new post manage extension plugin result with the provided
   * information.
   *
   * @param  restartServer         Indicates whether the manage-extension tool
   *                               should restart the server after the
   *                               installation if it was previous started.
   * @param  messages              Messages to print to the console or
   *                               <code>null</code>.
   */
  public PostManageExtensionPluginResult(final boolean restartServer,
                                         final List<String> messages)
  {
    this.restartServer = restartServer;
    this.messages = messages;
  }


  /**
   * Indicates whether the manage-extension tool should restart the server
   * after the installation if it was previous started.
   *
   * @return  {@code true} the manage-extension tool should restart
   * the server, or {@code false} if not.
   */
  public boolean restartServer()
  {
    return restartServer;
  }



  /**
   * Retrieves the list of messages to print to the console.
   *
   * @return  The list of messages to print to the console or <code>null</code>
   *          if no messages are to be printed.
   */
  public List<String> getMessages()
  {
    return messages;
  }


  /**
   * Retrieves a string representation of this post manage extension plugin
   * result.
   *
   * @return  A string representation of this post manage extension plugin
   * result.
   */
  @Override()
  public String toString()
  {
    final StringBuilder buffer = new StringBuilder();

    buffer.append("PostManageExtensionPluginResult(restartServer=");
    buffer.append(restartServer);
    buffer.append(", messages=");
    buffer.append(messages);
    buffer.append(')');

    return buffer.toString();
  }
}
