/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;



import java.io.File;
import java.io.InputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.security.GeneralSecurityException;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import com.unboundid.directory.sdk.ds.api.ChangeListener;
import com.unboundid.directory.sdk.common.api.DiskSpaceConsumer;
import com.unboundid.directory.sdk.common.api.MonitorProvider;
import com.unboundid.directory.sdk.common.api.ServerShutdownListener;
import com.unboundid.directory.sdk.common.api.ServerThread;
import com.unboundid.directory.sdk.common.config.GenericConfig;
import com.unboundid.directory.sdk.common.schema.Schema;
import com.unboundid.directory.sdk.ds.types.RegisteredChangeListener;
import com.unboundid.directory.sdk.proxy.types.Location;
import com.unboundid.ldap.sdk.ChangeType;
import com.unboundid.ldap.sdk.DN;
import com.unboundid.ldap.sdk.Filter;
import com.unboundid.ldap.sdk.LDAPConnection;
import com.unboundid.ldap.sdk.LDAPConnectionOptions;
import com.unboundid.ldap.sdk.LDAPConnectionPool;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.LDAPInterface;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface may be used to obtain information about the server in
 * which an extension is running.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface ServerContext
{
  /**
   * Retrieves the compact name of the server vendor.  The returned name will
   * not have any spaces.
   *
   * @return  The compact name of the server vendor.
   */
  String getShortVendorName();



  /**
   * Retrieves the full name of the server vendor.  The returned name may
   * contain spaces.
   *
   * @return  The full name of the server vendor.
   */
  String getFullVendorName();



  /**
   * Retrieves the compact name of the server.  The returned name will not have
   * any spaces.
   *
   * @return  The compact name of the server.
   */
  String getCompactProductName();



  /**
   * Retrieves the package name of the server that is used for defining the
   * server package ZIP file and root directory.  The returned name will not
   * have any spaces.
   *
   * @return  The package name of the server.
   */
  String getPackageName();



  /**
   * Retrieves the full name of the server.  The returned name may contain
   * spaces.
   *
   * @return  The full name of the server.
   */
  String getFullProductName();



  /**
   * Retrieves the base name of the server which is generally the full
   * product name without the vendor information.  The returned name may
   * contain spaces.
   *
   * @return  The full name of the server.
   */
  String getBasicProductName();



  /**
   * Retrieves the major version number for the server.  The major version
   * number is the first number which appears in the full version string (e.g.,
   * for a version string of "1.2.3.4-beta5", the major version number is 1).
   *
   * @return  The major version number for the server.
   */
  int getMajorVersionNumber();



  /**
   * Retrieves the minor version number for the server.  The minor version
   * number is the second number which appears in the full version string (e.g.,
   * for a version string of "1.2.3.4-beta5", the minor version number is 2).
   *
   * @return  The minor version number for the server.
   */
  int getMinorVersionNumber();



  /**
   * Retrieves the point version number for the server.  The point version
   * number is the third number which appears in the full version string (e.g.,
   * for a version string of "1.2.3.4-beta5", the point version number is 3).
   *
   * @return  The point version number for the server.
   */
  int getPointVersionNumber();



  /**
   * Retrieves the patch version number for the server.  The patch version
   * number is the fourth number which appears in the full version string (e.g.,
   * for a version string of "1.2.3.4-beta5", the patch version number is 4).
   *
   * @return  The point version number for the server.
   */
  int getPatchVersionNumber();



  /**
   * Retrieves a string that is the concatenation of the major, minor,
   * point, and patch version numbers of the server.
   *
   * @return the server's version number string.
   */
  String getVersionNumber();



  /**
   * Retrieves a string that is the concatenation of the product name, the
   * major, minor, point, and patch version numbers of the server and any
   * any other unique version information.
   *
   * @return the server's full version number string.
   */
  String getFullVersion();



  /**
   * Retrieves the version qualifier string for the server.  The version
   * qualifier is an optional string which may provide additional information
   * about the server version.  If a version qualifier is present, then it will
   * immediately follow the patch version number (e.g., for a version string of
   * "1.2.3.4-beta5", the version qualifier is "-beta5").
   *
   * @return  The version qualifier for the server, or an empty string if no
   *          version qualifier is defined.
   */
  String getVersionQualifier();



  /**
   * Retrieves a value which identifies the source revision (in the
   * version control system used to hold the server source code) from which the
   * server was built.
   *
   * @return  The source revision identifier for the server.
   */
  String getSourceRevision();



  /**
   * Indicates whether the server is in the process of starting up.
   *
   * @return  {@code true} if the server is in the process of starting up, or
   *          {@code false} if not.
   */
  boolean isStarting();



  /**
   * Indicates whether the server is currently running.
   *
   * @return  {@code true} if the server is running, or {@code false} if not.
   */
  boolean isRunning();



  /**
   * Indicates whether the server is in the process of shutting down.
   *
   * @return  {@code true} if the server is in the process of shutting down, or
   *          {@code false} if not.
   */
  boolean isShuttingDown();



  /**
   * Retrieves the time that the server was started.  The value returned will
   * be an offset in milliseconds since 12:00 a.m. on January 1, 1970.
   *
   * @return  The time that the server was started.
   */
  long getStartTime();



  /**
   * Retrieves a compact ID that was generated at the time the server was
   * started.  It is not guaranteed to be unique among all instances of the
   * server, but is guaranteed to be unique across all invocations of a single
   * server installation.  This is suitable for inclusion in log messages and
   * can help differentiate operations with the same connection ID and operation
   * ID across server restarts.
   *
   * @return  A compact ID that was generated at the time the server was
   *          started.
   */
  String getStartupID();



  /**
   * Retrieves a unique identifier that was generated at the time the server was
   * started.  It will be unique across all server instances and all invocations
   * of the same instance.
   *
   * @return  A unique identifier that was generated at the time the server was
   *          started.
   */
  UUID getStartupUUID();



  /**
   * Retrieves the instance name that has been assigned to the server.
   *
   * @return  The instance name that has been assigned to the server.
   */
  String getInstanceName();




  /**
    * Retrieves the location that has been assigned to the server, if any.
    *
    * @return  The location that has been assigned to the server, or
   *           {@code null} if no location has been assigned.
    */
   Location getLocation();



  /**
   * Retrieves the path to the server root directory.
   *
   * @return  The path to the server root directory.
   */
  File getServerRoot();



  /**
   * Retrieves a ToolExecutor that can be used to internally execute select
   * command line utilities.
   *
   * @return  A ToolExecutor that can be used to internally execute select
   *          command line utilities.
   */
  ToolExecutor getToolExecutor();



  /**
   * Indicates whether the extension is running in a server that has Directory
   * Server functionality available.
   *
   * @return  {@code true} if Directory Server functionality is available, or
   *          {@code false} if not.
   */
  boolean isDirectoryFunctionalityAvailable();



  /**
   * Indicates whether the extension is running in a server that has Directory
   * Proxy Server functionality available.
   *
   * @return  {@code true} if Directory Proxy Server functionality is available,
   *          or {@code false} if not.
   */
  boolean isDirectoryProxyFunctionalityAvailable();



  /**
   * Indicates whether the extension is running in a server that has
   * ${SYNC_SERVER_BASE_NAME} functionality available.
   *
   * @return  {@code true} if ${SYNC_SERVER_BASE_NAME} functionality is
   *          available, or {@code false} if not.
   */
  boolean isSyncFunctionalityAvailable();



  /**
   * Retrieves an internal connection that is authenticated as a root user
   * that is not subject to access control.
   * <BR><BR>
   * During operation processing, this connection is considered a root
   * connection that is both internal and secure.
   * <BR><BR>
   * Note that the returned connection will use the client connection policy
   * defined as the default policy for internal operations.  If you have access
   * to a {@code ClientContext} and wish to use the client connection policy
   * associated with that connection, use the
   * {@link ClientContext#getInternalRootConnection(boolean)}  method.
   *
   * @return  An internal connection that is authenticated as a root user.
   */
  InternalConnection getInternalRootConnection();



  /**
   * Retrieves an internal connection that is authenticated as the specified
   * user.  Operations on the connection may be subject to access control based
   * on the privileges associated with the specified user.
   * <BR><BR>
   * During operation processing, this connection is considered a user
   * connection that is both internal and secure.
   * <BR><BR>
   * Note that the returned connection will use the client connection policy
   * defined as the default policy for internal operations.  If you have access
   * to a {@code ClientContext} and wish to use the client connection policy
   * associated with that connection, use the
   * {@link ClientContext#getInternalConnection(String,boolean)} method.
   *
   * @param  dn  The DN of the user as whom the connection should be
   *             authenticated.  It may be {@code null} or empty if the
   *             connection should be unauthenticated.
   *
   * @return  An internal connection that is authenticated as the specified
   *          user.
   *
   * @throws  LDAPException  If a problem occurs while attempting to
   *                         authenticate as the specified user.
   */
  InternalConnection getInternalConnection(final String dn)
        throws LDAPException;



  /**
   * Retrieves a client connection that is authenticated as a root user
   * and is not subject to access control.
   * <BR><BR>
   * This method should be used when the connection will be used on behalf of
   * some external client, and allows you to indicate whether the external
   * client has a secure connection. This information is used by some of the
   * internal security controls, such as password policies, which may reject
   * password change operations if they are not over a secure connection, for
   * example.
   * <BR><BR>
   * During operation processing, this connection is considered a root
   * connection that is external and optionally secure, depending on the
   * {@code isClientSecure} parameter.
   *
   * @param  isClientSecure Whether the external client is connected over a
   *                        secure channel.
   *
   * @return  A client connection that is authenticated as a root user.
   */
  LDAPInterface getClientRootConnection(final boolean isClientSecure);



  /**
   * Retrieves a client connection that is authenticated as the specified
   * user.  Operations on the connection may be subject to access control based
   * on the privileges associated with the specified user.
   * <BR><BR>
   * This method should be used when the connection will be used on behalf of
   * some external client, and allows you to indicate whether the external
   * client has a secure connection. This information is used by some of the
   * internal security controls, such as password policies, which may reject
   * password change operations if they are not over a secure connection, for
   * example.
   * <BR><BR>
   * During operation processing, this connection is considered a user
   * connection that is external and optionally secure, depending on the
   * {@code isClientSecure} parameter.
   *
   * @param  dn  The DN of the user as whom the connection should be
   *             authenticated.  It may be {@code null} or empty if the
   *             connection should be unauthenticated.
   *
   * @param  isClientSecure Whether the external client is connected over a
   *                        secure channel.
   *
   * @return  An internal connection that is authenticated as the specified
   *          user.
   *
   * @throws  LDAPException  If a problem occurs while attempting to
   *                         authenticate as the specified user.
   */
  LDAPInterface getClientConnection(final String dn,
                                    final boolean isClientSecure)
       throws LDAPException;



  /**
   * Retrieves a reference to the server schema.
   *
   * @return  A reference to the server schema.
   */
  Schema getSchema();



  /**
   * Registers the provided OID with the server so that it will appear in the
   * supportedControl attribute of the server's root DSE.  This method should
   * only be used by extensions which implement some level of support for that
   * control (e.g., a pre-parse plugin which performs all necessary processing
   * from the control and removes it from the request so that it will not be
   * seen and potentially rejected by the core server).
   *
   * @param  oid  The OID to be registered.
   */
  void registerSupportedControlOID(final String oid);



  /**
   * Deregisters the provided OID with the server so that it will no longer
   * appear in the supportedControl attribute of the server's root DSE.
   *
   * @param  oid  The OID to be deregistered.
   */
  void deregisterSupportedControlOID(final String oid);



  /**
   * Registers the provided change listener with the server so that it may be
   * notified of any changes matching the provided criteria.
   *
   * @param  listener     The change listener to be registered with the server.
   *                      It must not be {@code null}.
   * @param  changeTypes  The types of changes for which the listener should be
   *                      registered.  It may be {@code null} or empty to
   *                      indicate that the listener should be registered for
   *                      all types of changes.
   * @param  baseDNs      The set of base DNs for which the listener should be
   *                      registered.  It may be {@code null} or empty to
   *                      indicate that the listener should be registered for
   *                      changes processed anywhere in the server.
   * @param  filter       A filter which may be used to restrict the set of
   *                      changes for which the listener is notified.  If a
   *                      filter is provided, then only changes in which the
   *                      target entry matches the given filter (either before
   *                      or after the change was processed) will be notified.
   *                      It may be {@code null} to indicate that the contents
   *                      of the entry should not be considered.
   *
   * @return  An object representing the change listener that has been
   *          registered with the server.
   *
   * @throws  LDAPException  If a problem is encountered while attempting to
   *                         register the provided change listener (e.g., if any
   *                         of the base DNs cannot be parsed as a valid DN).
   */
  RegisteredChangeListener registerChangeListener(final ChangeListener listener,
                                final Set<ChangeType> changeTypes,
                                final List<String> baseDNs,
                                final Filter filter)
       throws LDAPException;



  /**
   * Deregisters the provided change listener with the server.  This will have
   * no effect if the provided listener is not registered.
   *
   * @param  listener  The change listener to be deregistered.  It must not be
   *                   {@code null}.
   */
  void deregisterChangeListener(final RegisteredChangeListener listener);



  /**
   * Registers the provided disk space consumer with the server.
   *
   * @param  consumer  The disk space consumer to be registered with the server.
   *                   It must not be {@code null}.
   *
   * @return  An object representing the disk space consumer that has been
   *          registered with the server.
   */
  RegisteredDiskSpaceConsumer registerDiskSpaceConsumer(
                                   final DiskSpaceConsumer consumer);



  /**
   * Deregisters the provided disk space consumer with the server.  This will no
   * no effect if the provided consumer is not registered.
   *
   * @param  consumer  The disk space consumer to be deregistered with the
   *                   server. It must not be {@code null}.
   */
  void deregisterDiskSpaceConsumer(final RegisteredDiskSpaceConsumer consumer);



  /**
   * Registers the provided listener to be notified when the server shutdown
   * process has begun.
   *
   * @param  listener  The server shutdown listener to be registered.  It must
   *                   not be {@code null}.
   *
   * @return  An object representing the shutdown listener that has been
   *          registered with the server.
   */
  RegisteredServerShutdownListener registerShutdownListener(
                                        final ServerShutdownListener listener);



  /**
   * Deregisters the provided server shutdown listener with the server.  This
   * will have no effect if the provided listener is not registered.
   *
   * @param  listener  The server shutdown listener to be deregistered.  It must
   *                   not be {@code null}.
   */
  void deregisterShutdownListener(
            final RegisteredServerShutdownListener listener);



  /**
   * Registers the given monitor provider with the server. It is important that
   * this monitor provider has a unique instance name across all monitor
   * providers in the server. If there is already a monitor provider registered
   * with the same instance name, an {@link IllegalStateException} will be
   * thrown.
   * <p>
   * The generated monitor entry will have a DN in the following format:
   * <code>cn={monitor-instance-name} [from {extension-type}:{extension-name}],
   * cn=monitor</code> and it will contain three auto-generated attributes:
   * <code>ds-extension-monitor-name</code>, <code>ds-extension-type</code>, and
   * <code>ds-extension-name</code>. Note: the {extension-type} and
   * {extension-name} are from the extension which owns this ServerContext
   * instance, not from the given MonitorProvider object.
   * <p>
   * The specified monitor provider does not need any server-side configuration,
   * and the configuration-related methods
   * (<code>initalizeMonitorProvider()</code>,
   *  <code>finalizeMonitorProvider()</code>,
   *  <code>defineConfigArguments()</code>,
   *  <code>isConfigurationAcceptable()</code>,
   *  <code>applyConfiguration()</code>) do not need to be implemented because
   *  they will not be called.
   * <p>
   * When an extension is disabled, all of its registered monitor
   * providers will automatically deregister themselves from the server.
   * You can also manually deregister them using
   * {@link #deregisterMonitorProvider(RegisteredMonitorProvider)} or
   * {@link #deregisterAllMonitorProviders()}.
   *
   * @param provider The monitor provider instance to be registered. It
   *                 must not be {@code null}.
   * @param config The configuration object from the extension that is
   *               registering the given monitor provider. This is required so
   *               that the monitor entry can be given a unique DN which
   *               includes the name of the extension that registered it.
   *
   * @return An object representing the monitor provider that has been
   *         registered with the server.
   */
  RegisteredMonitorProvider registerMonitorProvider(
                                 final MonitorProvider provider,
                                 final GenericConfig config);



  /**
   * Deregisters the given monitor provider with the server. This will have no
   * effect if the given monitor provider is not registered.
   *
   * @param provider The monitor provider instance to be registered. It
   *                 must not be {@code null}.
   */
  void deregisterMonitorProvider(final RegisteredMonitorProvider provider);



  /**
   * Deregisters all the monitor providers that were registered with the server
   * by this instance of {@link ServerContext}. This can be useful during
   * cleanup or if you want to clear out all the existing monitor data from a
   * given extension and register new monitor providers.
   * <p>
   * This will be called automatically when your extension is unloaded or
   * disabled.
   */
  void deregisterAllMonitorProviders();



  /**
   * Creates a new thread to run within the server.  The thread will not be
   * started.
   *
   * @param  name          The name to use for the thread.  It must not be
   *                       {@code null} or empty.
   * @param  serverThread  The class providing the logic for the thread.  It
   *                       must not be {@code null}.
   *
   * @return  The thread that has been created but not yet started.
   */
  Thread createThread(final ServerThread serverThread, final String name);



  /**
   * Writes a message to the server error log.
   *
   * @param  severity  The severity to use for the log message.  It must not be
   *                   {@code null}.
   * @param  message   The message to be logged.  It must not be {@code null}.
   */
  void logMessage(final LogSeverity severity, final String message);



  /**
   * Generates an administrative alert notification.
   *
   * @param  severity  The severity to use for the alert notification.  It must
   *                   not be {@code null}.
   * @param  message   The message to be used for the alert notification.  It
   *                   must not be {@code null}.
   */
  void sendAlert(final AlertSeverity severity, final String message);



  /**
   * Generates an administrative alert notification.
   *
   * @param  alertTypeName  The name to use to identify the alert type.  Each
   *                        kind of alert must have a distinct name and all
   *                        alerts with this alert type must always be used with
   *                        the same severity and OID values.  Alert type names
   *                        must start with a lowercase ASCII letter and must
   *                        contain only lowercase ASCII letters, numeric
   *                        digits, and dashes.
   * @param  severity       The severity to use for the alert notification.
   *                        It must not be {@code null}.
   * @param  alertTypeOID   The numeric OID for the alert type.  It must not be
   *                        {@code null}, and it must be a valid numeric OID.
   *                        The same OID must always be used for the associated
   *                        alert type, and each different alert type must have
   *                        a unique OID.
   * @param  message        The message to be used for the alert notification.
   *                        It must not be {@code null}.
   *
   * @throws  LDAPException  If the provided information cannot be used to
   *                         generate a valid alert (e.g., if the alert type
   *                         name does not meet the naming constraints or has
   *                         already been used with a different severity and/or
   *                         OID, or if the OID has already been used with a
   *                         different alert type).
   */
  void sendAlert(final String alertTypeName, final AlertSeverity severity,
                 final String alertTypeOID, final String message)
       throws LDAPException;



  /**
   * Generates an administrative alert and updates the server's general monitor
   * entry to list the specified alert type as a degraded alert type.  Once the
   * condition that triggered the server to be considered degraded has been
   * resolved, the {@link DegradedAlertType#setResolved} method should be used
   * to remove the alert type from the list of degraded alert types.
   * <BR><BR>
   * If a server has one or more unavailable alert types, it will be considered
   * unavailable for use by products which make use of this information.  If a
   * server does not have any unavailable alert types but has one or more
   * degraded alert types, then it will be considered degraded and less
   * desirable for use by products which make use of this information.
   *
   * @param  alertTypeName  The name to use to identify the alert type.  Each
   *                        kind of alert must have a distinct name and all
   *                        alerts with this alert type must always be used with
   *                        the same severity and OID values.  Alert type names
   *                        must start with a lowercase ASCII letter and must
   *                        contain only lowercase ASCII letters, numeric
   *                        digits, and dashes.
   * @param  severity       The severity to use for the alert notification.
   *                        It must not be {@code null}.
   * @param  alertTypeOID   The numeric OID for the alert type.  It must not be
   *                        {@code null}, and it must be a valid numeric OID.
   *                        The same OID must always be used for the associated
   *                        alert type, and each different alert type must have
   *                        a unique OID.
   * @param  message        The message to be used for the alert notification.
   *                        It must not be {@code null}.
   *
   * @return  An object that may be used to obtain information about the alert
   *          type and indicate that the degraded condition has been resolved.
   *
   * @throws  LDAPException  If the provided information cannot be used to
   *                         generate a valid alert (e.g., if the alert type
   *                         name does not meet the naming constraints or has
   *                         already been used with a different severity and/or
   *                         OID, or if the OID has already been used with a
   *                         different alert type).
   */
  DegradedAlertType sendDegradedAlertNotification(final String alertTypeName,
                                                  final AlertSeverity severity,
                                                  final String alertTypeOID,
                                                  final String message)
       throws LDAPException;



  /**
   * Generates an administrative alert and updates the server's general monitor
   * entry to list the specified alert type as an unavailable alert type.  Once
   * the condition that triggered the server to be considered unavailable has
   * been resolved, the {@link UnavailableAlertType#setResolved} method should
   * be used to remove the alert type from the list of unavailable alert types.
   * <BR><BR>
   * If a server has one or more unavailable alert types, it will be considered
   * unavailable for use by products which make use of this information.  If a
   * server does not have any unavailable alert types but has one or more
   * degraded alert types, then it will be considered degraded and less
   * desirable for use by products which make use of this information.
   *
   * @param  alertTypeName  The name to use to identify the alert type.  Each
   *                        kind of alert must have a distinct name and all
   *                        alerts with this alert type must always be used with
   *                        the same severity and OID values.  Alert type names
   *                        must start with a lowercase ASCII letter and must
   *                        contain only lowercase ASCII letters, numeric
   *                        digits, and dashes.
   * @param  severity       The severity to use for the alert notification.
   *                        It must not be {@code null}.
   * @param  alertTypeOID   The numeric OID for the alert type.  It must not be
   *                        {@code null}, and it must be a valid numeric OID.
   *                        The same OID must always be used for the associated
   *                        alert type, and each different alert type must have
   *                        a unique OID.
   * @param  message        The message to be used for the alert notification.
   *                        It must not be {@code null}.
   *
   * @return  An object that may be used to obtain information about the alert
   *          type and indicate that the unavailable condition has been
   *          resolved.
   *
   * @throws  LDAPException  If the provided information cannot be used to
   *                         generate a valid alert (e.g., if the alert type
   *                         name does not meet the naming constraints or has
   *                         already been used with a different severity and/or
   *                         OID, or if the OID has already been used with a
   *                         different alert type).
   */
  UnavailableAlertType sendUnavailableAlertNotification(
                            final String alertTypeName,
                            final AlertSeverity severity,
                            final String alertTypeOID, final String message)
       throws LDAPException;



  /**
   * Indicates whether debugging is enabled in the server.
   * <BR><BR>
   * Note that debug messages written by Server SDK extensions will be handled
   * in the same way as debug messages generated in other areas of the server.
   * However, the "Server SDK Extension Debug Logger" may be used to easily
   * obtain only debug messages generated by Server SDK extensions, suppressing
   * all messages generated in other areas of the server.
   *
   * @return  {@code true} if debugging is enabled in the server, or
   *          {@code false} if not.
   */
  boolean debugEnabled();



  /**
   * Writes a debug message indicating that the provided exception has been
   * caught.
   * <BR><BR>
   * Note that debug messages written by Server SDK extensions will be handled
   * in the same way as debug messages generated in other areas of the server.
   * However, the "Server SDK Extension Debug Logger" may be used to easily
   * obtain only debug messages generated by Server SDK extensions, suppressing
   * all messages generated in other areas of the server.
   *
   * @param  t  The exception that has been caught.
   */
  void debugCaught(final Throwable t);



  /**
   * Writes a debug message indicating that the provided exception will be
   * thrown.
   * <BR><BR>
   * Note that debug messages written by Server SDK extensions will be handled
   * in the same way as debug messages generated in other areas of the server.
   * However, the "Server SDK Extension Debug Logger" may be used to easily
   * obtain only debug messages generated by Server SDK extensions, suppressing
   * all messages generated in other areas of the server.
   *
   * @param  t  The exception that will be thrown.
   */
  void debugThrown(final Throwable t);



  /**
   * Writes a debug message with an error severity.
   * <BR><BR>
   * Note that debug messages written by Server SDK extensions will be handled
   * in the same way as debug messages generated in other areas of the server.
   * However, the "Server SDK Extension Debug Logger" may be used to easily
   * obtain only debug messages generated by Server SDK extensions, suppressing
   * all messages generated in other areas of the server.
   *
   * @param  message  The message to be debugged.
   */
  void debugError(final String message);



  /**
   * Writes a debug message with a warning severity.
   * <BR><BR>
   * Note that debug messages written by Server SDK extensions will be handled
   * in the same way as debug messages generated in other areas of the server.
   * However, the "Server SDK Extension Debug Logger" may be used to easily
   * obtain only debug messages generated by Server SDK extensions, suppressing
   * all messages generated in other areas of the server.
   *
   * @param  message  The message to be debugged.
   */
  void debugWarning(final String message);



  /**
   * Writes a debug message with an informational severity.
   * <BR><BR>
   * Note that debug messages written by Server SDK extensions will be handled
   * in the same way as debug messages generated in other areas of the server.
   * However, the "Server SDK Extension Debug Logger" may be used to easily
   * obtain only debug messages generated by Server SDK extensions, suppressing
   * all messages generated in other areas of the server.
   *
   * @param  message  The message to be debugged.
   */
  void debugInfo(final String message);



  /**
   * Writes a debug message with a verbose severity.
   * <BR><BR>
   * Note that debug messages written by Server SDK extensions will be handled
   * in the same way as debug messages generated in other areas of the server.
   * However, the "Server SDK Extension Debug Logger" may be used to easily
   * obtain only debug messages generated by Server SDK extensions, suppressing
   * all messages generated in other areas of the server.
   *
   * @param  message  The message to be debugged.
   */
  void debugVerbose(final String message);



  /**
   * Retrieves a connection that is established and (if appropriate)
   * authenticated to the specified LDAP External server.  This will be a
   * newly-created connection that is not used for any other purpose, and the
   * caller is responsible for closing that connection when it is no longer
   * needed.
   *
   * @param  ldapExternalServerCfgObjectName  The name of the configuration
   *                                          object for the LDAP external
   *                                          server to which the connection
   *                                          should be established.  It must
   *                                          not be {@code null}.
   * @param  connectionOptions                The set of options to use for the
   *                                          connection that is established.
   *                                          It may be {@code null} if a
   *                                          default set of options should be
   *                                          used.
   *
   * @return  The newly-created LDAP connection.
   *
   * @throws  LDAPException  If it is not possible to establish a connection to
   *                         the target servers.
   */
  LDAPConnection getLDAPExternalServerConnection(
                      final String ldapExternalServerCfgObjectName,
                      final LDAPConnectionOptions connectionOptions)
       throws LDAPException;



  /**
   * Retrieves a connection pool with connections to the specified server.  This
   * will be a newly-created connection pool that is not used for any other
   * purpose, and the caller is responsible for closing that connection pool
   * when it is no longer needed.
   *
   * @param  ldapExternalServerCfgObjectName  The name of the configuration
   *                                          object for the LDAP external
   *                                          server to which the connections
   *                                          should be established.  It must
   *                                          not be {@code null}.
   * @param  connectionOptions                The set of options to use for the
   *                                          connection that is established.
   *                                          It may be {@code null} if a
   *                                          default set of options should be
   *                                          used.
   * @param  initialConnections               The initial number of connections
   *                                          to attempt to establish.  It must
   *                                          be greater than or equal to zero.
   * @param  maxConnections                   The maximum number of connections
   *                                          that should be established and
   *                                          unused in the pool at any time.
   *                                          It must be greater than or equal
   *                                          to {@code initialConnections}, and
   *                                          it must not be zero.
   * @param  throwOnConnectFailure            Indicates whether to throw an
   *                                          {@code LDAPException} if an error
   *                                          is encountered while attempting to
   *                                          connect or authenticate any of the
   *                                          initial connections.  If this is
   *                                          {@code false} and none of the
   *                                          initial connections can be
   *                                          established (or if the initial
   *                                          number of connections is zero),
   *                                          then the pool will be returned
   *                                          without any connections.
   *
   * @return  The newly-created LDAP connection pool.
   *
   * @throws  LDAPException  If it is not possible to establish a pool of
   *                         connections to the specified LDAP external server.
   */
  LDAPConnectionPool getLDAPExternalServerConnectionPool(
                          final String ldapExternalServerCfgObjectName,
                          final LDAPConnectionOptions connectionOptions,
                          final int initialConnections,
                          final int maxConnections,
                          final boolean throwOnConnectFailure)
       throws LDAPException;



  /**
   * Retrieves the plaintext representation of the specified obscured value from
   * the server configuration.  Obscured values may be used to allow extensions
   * to store potentially sensitive information (for example, credentials needed
   * to access some external system) in the configuration in a manner that will
   * prevent them from being stored in the clear or displayed in the clear in
   * administrative interfaces.  If a server extension needs to be configured
   * with potentially sensitive information, it is recommended that each piece
   * of sensitive information be stored as an obscured value, and that the
   * extension be configured with the names of the relevant obscured values.
   *
   * @param  obscuredValueConfigObjectName  The name of the configuration object
   *                                        for the obscured value to retrieve.
   *                                        It must not be {@code null}.
   *
   * @return  The plaintext representation of the specified obscured value from
   *          the server configuration.
   *
   * @throws  LDAPException  If the provided name does not reference an obscured
   *                         value that is defined in the server configuration.
   */
  String getObscuredValue(final String obscuredValueConfigObjectName)
         throws LDAPException;



  /**
   * Returns a ValueConstructor built from the provided value template. A
   * ValueConstructor can be used to construct string values from a template
   * that can reference attributes on an entry.
   *
   * @param  constructedValueTemplate  The constructed value pattern. See
   *                                   {@link ValueConstructor} for details on
   *                                   the syntax for the template.
   *
   * @return  A ValueConstructor based on the provided template.
   *
   * @throws  LDAPException  If the provided value template is invalid.
   */
  ValueConstructor createValueConstructor(final String constructedValueTemplate)
        throws LDAPException;



  /**
   * Retrieves the group defined in the specified entry.
   *
   * @param  groupDN  The DN of the entry containing the group to retrieve.  It
   *                  must not be {@code null}.
   *
   * @return  The group defined in the specified entry.
   *
   * @throws  LDAPException  If the specified entry does not exist, if it exists
   *                         but does not represent a group, or if a problem is
   *                         encountered while retrieving the group.
   */
  Group getGroup(final String groupDN)
        throws LDAPException;



  /**
   * Retrieves the set of groups in which the specified user is a member,
   * indexed by group DN.
   *
   * @param  userDN                The DN of the user for whom to retrieve the
   *                               associated groups.  It must not be
   *                               {@code null}.
   * @param  directMembershipOnly  Indicates whether to only consider the user a
   *                               member of the group if that user is directly
   *                               named as a member of the group.  If this
   *                               parameter is {@code true}, then the method
   *                               will return {@code true} only if the user is
   *                               directly listed as a member of the group, and
   *                               will not included dynamic or nested
   *                               membership.  If this parameter is
   *                               {@code false}, then the method will return
   *                               {@code true} if the user is a direct member
   *                               of the group, is a member of a dynamic group
   *                               in which the user matches the membership
   *                               criteria, or if the user is a member of a
   *                               nested group.
   *
   * @return  A map of the groups in which the specified user is a member, or an
   *          empty map if the specified user is not a member of any groups.
   *
   * @throws  LDAPException  If the specified user does not exist or if a
   *                         problem is encountered while attempting to retrieve
   *                         the groups in which the user is a member.
   */
  Map<DN,Group> getGroupsForUser(final String userDN,
                                 final boolean directMembershipOnly)
                throws LDAPException;



  /**
   * Indicates whether the specified user has the indicated privilege.  The set
   * of defined privileges may be found in the privilege-list.html and
   * privilege-list.csv files in the server docs directory.
   *
   * @param  userDN         The DN of the user for whom to make the
   *                        determination.  It must not be {@code null}.
   * @param  privilegeName  The name of the privilege for which to make the
   *                        determination.  It must not be {@code null}.
   *
   * @return  {@code true} if the specified user has the indicated privilege, or
   *          {@code false} if not.
   *
   * @throws  LDAPException  If the specified user does not exist, if the
   *                         indicated privilege is not defined in the server,
   *                         or if a problem is encountered while trying to make
   *                         the determination.
   */
  boolean hasPrivilege(final String userDN, final String privilegeName)
          throws LDAPException;



  /**
   * Retrieves the names of the privileges held by the specified user.  The set
   * of defined privileges may be found in the privilege-list.html and
   * privilege-list.csv files in the server docs directory.
   *
   * @param  userDN  The DN of the user for whom to retrieve the privileges.  It
   *                  must not be {@code null}.
   *
   * @return  The names of the privileges held by the specified user, or an
   *          empty set if the user does not have any privileges.
   *
   * @throws  LDAPException  If the specified user does not exist, or if a
   *                         problem is encountered while attempting to retrieve
   *                         the user's privileges.
   */
  Set<String> getPrivilegeNames(final String userDN)
              throws LDAPException;



  /**
   * Reload the key and trust manager providers associated with any HTTP
   * connection handlers.that provide support for HTTPS.  This can cause any
   * changes to the certificates in those stores to be applied to the running
   * instance without the need to disable and re-enable the connection handler
   * or to restart the server. Note that this reload does have the potential
   * to interfere with the ability for clients to resume a TLS session created
   * before the reload was invoked.
   *
   * @throws LDAPException  If an error occurs.
   */
  void reloadHTTPConnectionHandlerCertificates()
       throws LDAPException;



  /**
   * Opens an input stream that may be used to read data from the provided
   * file.  The contents of the file may optionally have been encrypted with the
   * LDAP SDK's {@code PassphraseEncryptedOutputStream} using a key from the
   * server's encryption settings database, and may have optionally been
   * compressed with the GZIP compression algorithm.  If the file is both
   * compressed and encrypted, then the data must have been compressed before
   * it was encrypted.
   *
   * @param  file  The file to be read.  It must not be {@code null}.
   *
   * @return  The input stream that may be used to read the optionally
   *          compressed and/or encrypted file.
   *
   * @throws  LDAPException  If a problem is encountered while interacting with
   *                         the server's encryption framework, or if the data
   *                         is encrypted with a key not contained in the
   *                         encryption settings database.
   *
   * @throws  GeneralSecurityException  If a problem is encountered while
   *                                    preparing to decrypt the data.
   *
   * @throws  IOException  If a problem is encountered while attempting to read
   *                       or decompress the data.
   */
  InputStream getPossiblyCompressedAndEncryptedInputStream(final File file)
              throws LDAPException, GeneralSecurityException, IOException;



  /**
   * Examines the provided input stream and determines whether its contents have
   * been encrypted with a {@code PassphraseEncryptedOutputStream}, and if so,
   * retrieves an input stream that may be used to read its decrypted content.
   * Also examines the input stream to determine whether its contents may have
   * been GZIP-compressed, and if so, retrieves an input stream that may be used
   * to read its decompressed content.  If the data is encrypted, it must start
   * with a {@code PassphraseEncryptedStreamHeader}, and it must have been
   * encrypted with a key contained in the server's encryption settings
   * database.  If the data is both compressed and encrypted, then it must have
   * been compressed before it was encrypted.
   * <BR><BR>
   * Note that if a problem occurs that causes this method to throw an
   * exception, then the provided stream will have been closed before control
   * returns to the caller.
   *
   * @param  inputStream  The input stream from which the data is to be
   *                      obtained.  It must not be {@code null}.
   *
   * @return  The input stream that may be used to read the optionally
   *          compressed and/or encrypted data from the provided input stream.
   *
   * @throws  LDAPException  If a problem is encountered while interacting with
   *                         the server's encryption framework, or if the data
   *                         is encrypted with a key not contained in the
   *                         encryption settings database.
   *
   * @throws  GeneralSecurityException  If a problem is encountered while
   *                                    preparing to decrypt the data.
   *
   * @throws  IOException  If a problem is encountered while attempting to read
   *                       or decompress the data.
   */
  InputStream getPossiblyCompressedAndEncryptedInputStream(
                   final InputStream inputStream)
              throws LDAPException, GeneralSecurityException, IOException;



  /**
   * Wraps the provided output stream in a
   * {@code PassphraseEncryptedOutputStream} so that data written to it will be
   * encrypted, optionally after also wrapping it in a {@code GZIPOutputStream}
   * to ensure that the data is compressed before it is encrypted.  The
   * encryption key will be obtained from the server's preferred encryption
   * settings definition.
   *
   * @param  outputStream  The output stream to be wrapped.  It must not be
   *                       {@code null}.
   * @param  compress      Indicates whether to GZIP-compress the data before
   *                       encrypting it.
   *
   * @return  The output stream that may be used to write encrypted (and
   *          possibly compressed) data.
   *
   * @throws  LDAPException  If a problem is encountered while interacting with
   *                         the server's encryption framework or if the server
   *                         encryption settings database does not have a
   *                         preferred definition.
   *
   * @throws  GeneralSecurityException  If a problem is encountered while
   *                                    preparing to encrypt the data.
   *
   * @throws  IOException  If a problem occurs while preparing to compress the
   *                       the data.
   */
  OutputStream getEncryptedOutputStream(final OutputStream outputStream,
                                        final boolean compress)
               throws LDAPException, IOException, GeneralSecurityException;



  /**
   * Wraps the provided output stream in a
   * {@code PassphraseEncryptedOutputStream} so that data written to it will be
   * encrypted, optionally after also wrapping it in a {@code GZIPOutputStream}
   * to ensure that the data is compressed before it is encrypted.
   *
   * @param  outputStream                    The output stream to be wrapped.
   *                                         It must not be {@code null}.
   * @param  encryptionSettingsDefinitionID  The identifier of the encryption
   *                                         settings definition that should be
   *                                         used to encrypt the data.  It must
   *                                         not be {@code null}.
   * @param  compress                        Indicates whether to GZIP-compress
   *                                         the data before encrypting it.
   *
   * @return  The output stream that may be used to write encrypted (and
   *          possibly compressed) data.
   *
   * @throws  LDAPException  If a problem is encountered while interacting with
   *                         the server's encryption framework, or if the
   *                         encryption settings database does not have a
   *                         definition with the specified ID.
   *
   * @throws  GeneralSecurityException  If a problem is encountered while
   *                                    preparing to encrypt the data.
   *
   * @throws  IOException  If a problem occurs while preparing to compress the
   *                       the data.
   */
  OutputStream getEncryptedOutputStream(final OutputStream outputStream,
                    final String encryptionSettingsDefinitionID,
                    final boolean compress)
               throws LDAPException, GeneralSecurityException, IOException;



  /**
   * Retrieves information about the server's preferred encryption settings
   * definition, which is the default definition that will be used for new
   * encryption operations if no other definition is requested.
   *
   * @return  Information about the server's preferred encryption settings
   *          definition, or {@code null} if the server does not have a
   *          preferred encryption settings definition.
   */
  EncryptionSettingsDefinition getPreferredEncryptionSettingsDefinition();



  /**
   * Retrieves the encryption settings definition with the specified ID.
   *
   * @param  id  The identifier for the encryption settings definition to
   *             retrieve.  It must not be {@code null}.
   *
   * @return  The encryption settings definition with the specified ID, or
   *          {@code null} if the encryption settings database does not have a
   *          definition with the specified ID.
   */
  EncryptionSettingsDefinition getEncryptionSettingsDefinition(final String id);



  /**
   * Retrieves a list of all of the encryption settings definitions contained
   * in the server's encryption settings database.
   *
   * @return  A list of all the encryption settings definitions contained in
   *          the server's encryption settings database.
   */
  List<EncryptionSettingsDefinition> getEncryptionSettingsDefinitions();
}
