/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2016-2019 Ping Identity Corporation
 */


package com.unboundid.directory.sdk.common.types;

import com.unboundid.util.NotExtensible;

import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * Object returned from a request to an AccessTokenValidator to validate a
 * token. The fields of this object are derived from the IETF OAuth 2.0 Token
 * Introspection specification (RFC7662).
 * <p>
 * The decision as to whether an access token is accepted or not varies by
 * product. For the ${DIRECTORY_SERVER_BASE_NAME}, ${PROXY_SERVER_BASE_NAME},
 * ${SYNC_SERVER_BASE_NAME}, or ${METRICS_ENGINE_BASE_NAME}, different REST APIs
 * may enforce different authorization rules, depending on their configuration;
 * please refer to the relevant documentation. In general, a token's
 * {@code active} property must be true and its {@code tokenOwner} must not be
 * null.
 * <p>
 * For the ${BROKER_BASE_NAME}, authorization decisions are made by the policy
 * engine. All properties of the TokenValidationResult may be accessed by
 * policies; however, the default policy may only examine a subset of those
 * properties (e.g. especially the {@code active} property).
 */
@NotExtensible()
public final class TokenValidationResult {


  private final Boolean active;

  private final Set<String> scope;

  private final String clientId;

  private final String username;

  private final String tokenType;

  private final Long expirationTime;

  private final Long issuedAt;

  private final Long notUsedBefore;

  private final String tokenSubject;

  private final String tokenOwner;

  private final Set<String> audience;

  private final String tokenIdentifier;

  /**
   * This field contains any extension values implemented by the
   * Token Validator.
   */
  private final Map<String,Object> additionalProperties;

  /**
   * Private constructor for builder.
   * @param builder Builder source for properties.
   */
  private TokenValidationResult(final Builder builder)
  {
    this.active = builder.active;
    this.scope = builder.scope;
    this.clientId = builder.clientId;
    this.username = builder.username;
    this.tokenType = builder.tokenType;
    this.expirationTime = builder.expirationTime;
    this.issuedAt = builder.issuedAt;
    this.notUsedBefore = builder.notUsedBefore;
    this.tokenSubject = builder.tokenSubject;
    this.tokenOwner = builder.tokenOwner;
    this.audience = builder.audience;
    this.tokenIdentifier = builder.tokenIdentifier;
    this.additionalProperties = (builder.additionalProperties == null) ?
        Collections.<String,Object>emptyMap() :
        builder.additionalProperties;
  }

  /**
   * Get whether the token is active. Per RFC 7662, a value of true for this
   * property indicates that the given token was issued by the Token
   * Validator's associated authorization server, has not been revoked by the
   * resource owner, and is within its given time window of validity
   * (e.g. after its issuance time and before its expiration time).
   * @return the active state.
   */
  public Boolean getActive() {
    return active;
  }

  /**
   * Get the scopes granted to this token.
   * @return set of scope names.
   */
  public Set<String> getScope() {
    return scope;
  }

  /**
   * Get the OAuth2 client Id of the client that requested the token.
   * @return client Id string.
   */
  public String getClientId() {
    return clientId;
  }

  /**
   * Get a human-readable user name for the resource owner that authorized
   * the token.
   * @return user name string.
   */
  public String getUsername() {
    return username;
  }

  /**
   * Get the token type.  The type of the token is defined in section 5.1 of
   * OAuth 2.0 [RFC6749].
   * @return token type string.
   */
  public String getTokenType() {
    return tokenType;
  }

  /**
   * Get the token expiration time in seconds since January 1 1970 UTC.
   * @return expiration time in seconds.
   */
  public Long getExpirationTime() {
    return expirationTime;
  }

  /**
   * Get the token issue time in seconds since January 1 1970 UTC.
   * @return token issue time in seconds.
   */
  public Long getIssuedAt() {
    return issuedAt;
  }

  /**
   * Get the not used before time in seconds since January 1 1070 UTC.
   * @return not used before time in seconds.
   */
  public Long getNotUsedBefore() {
    return notUsedBefore;
  }

  /**
   * Get the token subject as defined in JWT [RFC7519].  This value should
   * contain the subject ID as known to the external authorization server rather
   * than the local server.
   * @return token subject string.
   */
  public String getTokenSubject() {
    return tokenSubject;
  }

  /**
   * Get the token subject as defined in JWT [RFC7519].  This value should
   * contain the subject ID as known to the ${SHORT_VENDOR_NAME} server rather
   * than the external authorization server.
   * <p>
   * It is the responsibility of the Token Validator to translate an external
   * subject identifier into the local server's syntax.
   * This syntax varies by product. For the ${DIRECTORY_SERVER_BASE_NAME},
   * ${PROXY_SERVER_BASE_NAME}, ${SYNC_SERVER_BASE_NAME}, or
   * ${METRICS_ENGINE_BASE_NAME}, the value should be an LDAP DN.
   * The ${BROKER_BASE_NAME} uses a relative resource path as the subject ID,
   * e.g. "Users/user_id", where user_id is the value of the subject's SCIM
   * id property.
   * @return token owner string.
   */
  public String getTokenOwner() {
    return tokenOwner;
  }

  /**
   * Get the token's intended audience(s).  The audience is a Service-specific
   * identifier representing the intended audience(s) for this token, as
   * defined in JWT [RFC7519].
   * @return token audience.
   */
  public Set<String> getAudience() {
    return audience;
  }

  /**
   * Get the unique identifier for this token. as defined in section 4.1.7 of
   * JWT [RFC7519].
   * @return token unique identifier string.
   */
  public String getTokenIdentifier() {
    return tokenIdentifier;
  }

  /**
   * Get any extension properties associated with the token.
   * @return Map of property names to values.
   */
  public Map<String, Object> getAdditionalProperties() {
    return additionalProperties;
  }

  /**
   * A Builder for TokenValidationResult.
   */
  public static class Builder
  {
    private Boolean active;
    private Set<String> scope;
    private String clientId;
    private String username;
    private String tokenType;
    private Long expirationTime;
    private Long issuedAt;
    private Long notUsedBefore;
    private String tokenSubject;
    private String tokenOwner;
    private Set<String> audience;
    private String tokenIdentifier;
    private Map<String,Object> additionalProperties;


    /**
     * Creates a new builder object.
     * @param active true if the token is active, false if not.  Per RFC 7662,
     *               a value of true for this property indicates that the given
     *               token was issued by the Token Validator's associated
     *               authorization server, has not been revoked by the resource
     *               owner, and is within its given time window of validity
     *               (e.g. after its issuance time and before its expiration
     *               time).
     */
    public Builder(final boolean active)
    {
      this.active = active;
      setScope(Collections.<String>emptySet());
    }

    /**
     * Set the active field of the token validation result.
     *
     * @param active true if the token is active, false if not.  Per RFC 7662,
     *               a value of true for this property indicates that the given
     *               token was issued by the Token Validator's associated
     *               authorization server, has not been revoked by the resource
     *               owner, and is within its given time window of validity
     *               (e.g. after its issuance time and before its expiration
     *               time).
     * @return this.
     */
    public Builder setActive(final boolean active)
    {
      this.active = active;
      return this;
    }

    /**
     * Set the optional scope field of the token validation result.
     *
     * @param scope A list of strings representing the scopes associated with
     *              this token.
     * @return this.
     */
    public Builder setScope(final Set<String> scope)
    {
      this.scope = (scope == null) ? Collections.<String>emptySet() : scope;
      return this;
    }

    /**
     * Set the optional clientId field of the token validation result.
     *
     * @param clientId Client identifier for the OAuth 2.0 client that
     *                 requested this token.
     * @return this.
     */
    public Builder setClientId(final String clientId)
    {
      this.clientId = clientId;
      return this;
    }

    /**
     * Set the optional username field of the token validation result.
     *
     * @param username Human-readable identifier for the resource owner who
     *                 authorized this token.
     * @return this.
     */
    public Builder setUsername(final String username)
    {
      this.username = username;
      return this;
    }

    /**
     * Set the optional tokenType field of the token validation result.
     *
     * @param tokenType Type of the token as defined in section 5.1 of OAuth
     *                  2.0 [RFC6749].
     * @return this.
     */
    public Builder setTokenType(final String tokenType)
    {
      this.tokenType = tokenType;
      return this;
    }

    /**
     * Set the optional expirationTime field of the token validation result.
     *
     * @param expirationTime Integer timestamp, measured in the number of
     *                       time units since January 1 1970 UTC, indicating
     *                       when this token will expire.
     * @param timeUnit the time units of the specified expiration time.
     * @return this.
     */
    public Builder setExpirationTime(
        final long expirationTime,
        final TimeUnit timeUnit)
    {
      this.expirationTime = timeUnit.toSeconds(expirationTime);
      return this;
    }

    /**
     * Set the optional issuedAt field of the token validation result.
     *
     * @param issuedAt Integer timestamp, measured in the number of time
     *                 units since January 1 1970 UTC, indicating when this
     *                 token was originally issued.
     * @param timeUnit the time units of the specified issuedAt time.
     * @return this.
     */
    public Builder setIssuedAt(
        final long issuedAt,
        final TimeUnit timeUnit)
    {
      this.issuedAt = timeUnit.toSeconds(issuedAt);
      return this;
    }

    /**
     * Set the optional notUsedBefore field of the token validation result.
     *
     * @param notUsedBefore Integer timestamp, measured in the number of
     *                      time units since January 1 1970 UTC, indicating
     *                      when this token is not to be used before.
     * @param timeUnit the time units of the specified notUsedBefore time.
     * @return this.
     */
    public Builder setNotUsedBefore(
        final long notUsedBefore,
        final TimeUnit timeUnit)
    {
      this.notUsedBefore = timeUnit.toSeconds(notUsedBefore);
      return this;
    }

    /**
     * Set the optional subjectToken field of the token validation result.
     *
     * @param tokenSubject Subject of the token, as defined in JWT [RFC7519].
     *                     This value should contain the subject ID as known to
     *                     the external authorization server rather than the
     *                     local server.
     * @return this.
     */
    public Builder setTokenSubject(final String tokenSubject)
    {
      this.tokenSubject = tokenSubject;
      return this;
    }

    /**
     * Set the optional tokenOwner field of the token validation result.
     *
     * @param tokenOwner Mapped subject of the token.
     *                   This value should contain the subject ID as known to
     *                   the local server rather than the external
     *                   authorization server.
     *                   <p>
     *                   It is the responsibility of the Token Validator to
     *                   translate an external subject identifier into the
     *                   local server's syntax. This syntax varies by product.
     *                   For the ${DIRECTORY_SERVER_BASE_NAME},
     *                   ${PROXY_SERVER_BASE_NAME}, ${SYNC_SERVER_BASE_NAME}, or
     *                   ${METRICS_ENGINE_BASE_NAME}, the value should be an
     *                   LDAP DN. The ${BROKER_BASE_NAME} uses a relative
     *                   resource path as the subject ID,  e.g. "Users/user_id",
     *                   where user_id is the value of the subject's SCIM id
     *                   property.
     * @return this.
     */
    public Builder setTokenOwner(final String tokenOwner)
    {
      this.tokenOwner = tokenOwner;
      return this;
    }

    /**
     * Set the optional audience field of the token validation result.
     *
     * @param audience Service-specific identifiers representing the
     *                 intended audience(s) for this token, as defined in
     *                 JWT [RFC7519].
     * @return this.
     */
    public Builder setAudience(final Set<String> audience)
    {
      this.audience = audience;
      return this;
    }

    /**
     * Set the optional tokenIdentifier field of the token validation result.
     *
     * @param tokenIdentifier Unique string identifier for the token, as
     *                        defined in section 4.1.7 of JWT [RFC7519].
     * @return this.
     */
    public Builder setTokenIdentifier(final String tokenIdentifier)
    {
      this.tokenIdentifier = tokenIdentifier;
      return this;
    }

    /**
     * Set the optional additionalProperties field of the token validation
     * result.
     * @param properties Map of property names to values, containing any
     *                   extension values implemented by the
     *                   AccessTokenValidator.
     * @return this.
     */
    public Builder setAdditionalProperties(
        final Map<String,Object> properties) {

      this.additionalProperties = properties;
      return this;
    }

    /**
     * Builds a new TokenValidationResult from the parameters
     * previously supplied to the builder.
     *
     * @return a new TokenValidationResult.
     */
    public TokenValidationResult build()
    {
      return new TokenValidationResult(this);
    }
  }

}
