/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;



import java.util.Set;

import com.unboundid.directory.sdk.common.schema.AttributeType;
import com.unboundid.ldap.sdk.Attribute;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.DN;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods which may be used to update the
 * contents of an entry.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_NOT_THREADSAFE)
public interface UpdatableEntry
       extends Entry
{
  /**
   * Specifies the DN for the entry.
   *
   * @param  dn  The DN for the entry.  It must not be {@code null}.
   *
   * @throws LDAPException  If the provided DN is not valid.
   */
  void setDN(final String dn)
       throws LDAPException;



  /**
   * Specifies the DN for the entry.
   *
   * @param  dn  The DN for the entry.  It must not be {@code null}.
   *
   * @throws LDAPException  If the provided DN is not valid.
   */
  void setDN(final DN dn)
       throws LDAPException;



  /**
   * Replaces the specified attribute in the entry, or adds the attribute if
   * it does not exist.
   *
   * @param  attribute  The attribute to be stored in the entry.  It must not be
   *                    {@code null}.
   */
  void setAttribute(final Attribute attribute);



  /**
   * Adds the provided attribute to the entry.  If an attribute already exists
   * with the same name and set of options, then the values will be merged.
   *
   * @param  attribute  The attribute to be added.  It must not be {@code null}.
   */
  void addAttribute(final Attribute attribute);



  /**
   * Removes all occurrences of the specified attribute from the entry.
   *
   * @param  type  The attribute type for the attribute to remove from the
   *               entry.  It must not be {@code null}.
   */
  void removeAttribute(final AttributeType type);



  /**
   * Removes all occurrences of the attribute with the specified name from the
   * entry.
   *
   * @param  name  The name or OID of the attribute to remove from the entry.
   *               It must not be {@code null}.
   */
  void removeAttribute(final String name);



  /**
   * Removes the attribute with the specified name and exact set of options
   * from the entry.
   *
   * @param  type     The attribute type for the attribute to remove from the
   *                  entry.  It must not be {@code null}.
   * @param  options  The set of attribute options for the attribute to remove
   *                  from the entry.  It may be {@code null} or empty if there
   *                  are no options.
   */
  void removeAttribute(final AttributeType type, final Set<String> options);



  /**
   * Removes the attribute with the specified name and exact set of options
   * from the entry.
   *
   * @param  name     The name or OID of the attribute to remove from the entry.
   *                  It must not be {@code null}.
   * @param  options  The set of attribute options for the attribute to remove
   *                  from the entry.  It may be {@code null} or empty if there
   *                  are no options.
   */
  void removeAttribute(final String name, final Set<String> options);
}
