/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2017-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.common.types;

import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;

import java.util.List;

/**
 * A ValueConstructor is used to build String values using a value-pattern
 * template that references attribute values within an Entry. An instance is
 * created by calling {@link ServerContext#createValueConstructor}.
 *
 * <h2>Referencing Attribute Values</h2>
 *
 * Any text provided in the value-pattern is copied verbatim into the
 * constructed value with the exception of attribute names embedded
 * in '{}'s. For instance, a value template of "{uid}@example.com"
 * and an entry with a uid value of "jsmith" would produce a value
 * of "jsmith@example.com".
 *
 * <p>
 * If a value template references multiple attributes, at most one may be
 * multivalued. If one attribute is multivalued then the returned value
 * will be multivalued with the same number of values.
 * For example a template of '{givenName} {sn}' would produce
 * values ['Jim Smith', 'James Smith'] if givenName had values of ['Jim',
 * 'James'] and sn had a value of 'Smith'. If the source entry has no
 * value or if there are multiple multivalued attributes, the attribute
 * value construction will fail.
 *
 * <p>
 * To include a '{' in the constructed value (when constructing JSON for
 * instance) use '{{' in the value-pattern. Likewise, '}}' will result in '}'
 * in the value pattern.
 *
 * <h2>Regular Expressions</h2>
 *
 * A regular expression with a substitution pattern can be used
 * to further manipulate an individual attribute
 * value.  The java.util.regex.Pattern and java.util.regex.Matcher
 * classes are used to perform the substitution.
 * The regular expression with replacement value uses this syntax
 * '{attr-name:/regex/replacement/flags}'.
 * <ul>
 *  <li>
 *   'attr-name' is an attribute name such as 'cn'.
 *  </li>
 *  <li>
 *   'regex' is a regular expression using the syntax described in the
 *   java.util.regex.Pattern javadoc.  The optional flag value can also
 *   be used to change how this regular expression is interpreted.
 *  </li>
 *  <li>
 *   'replacement' is a replacement value that is passed to the
 *   java.util.regex.Matcher#replaceFirst or java.util.regex.Matcher#replaceAll
 *   method to perform the replacement.  The 'g' (global) flag controls
 *   whether replaceFirst or replaceAll is called.
 *  </li>
 *  <li>
 *   'flags' is an optional set of single character flags that
 *   control how the regular expression is interpreted and how the replacement
 *   is performed. The flags are:
 *   <ul>
 *    <li>
 *     'g' : Matcher#replaceAll is called instead of Matcher#replaceFirst to
 *           replace all matching values instead of just the first one.
 *    </li>
 *    <li>
 *     'i' : the regex pattern is compiled with Pattern.CASE_INSENSITIVE.
 *    </li>
 *    <li>
 *     'x' : the regex pattern is compiled with Pattern.COMMENTS.
 *    </li>
 *    <li>
 *     's' : the regex pattern is compiled with Pattern.DOTALL.
 *    </li>
 *    <li>
 *     'm' : the regex pattern is compiled with Pattern.MULTILINE.
 *    </li>
 *    <li>
 *     'u' : the regex pattern is compiled with Pattern.UNICODE_CASE.
 *    </li>
 *    <li>
 *     'd' : the regex pattern is compiled with Pattern.UNIX_LINES.
 *    </li>
 *   </ul>
 * </ul>
 *
 * The substitution value is constructed using Java code equivalent to:
 * <pre>
 *   Pattern pattern = Pattern.compile(regex, flagsMask);
 *   Matcher matcher = pattern.matcher(attributeValueInput);
 *   String substitutionValueOutput = matcher.replaceFirst(replacement);
 * </pre>
 *
 * <h2>JSON Attribute Values</h2>
 *
 * To extract JSON fields within JSON attributes append '.' and then the
 * JSON field to extract to the attribute name. For example, if
 * 'ubidEmailJSON' is a JSON attribute and the 'value' field is to be
 * extracted then 'ubidEmailJSON.value' could be specified for the attribute
 * name resulting in '{ubidEmailJSON.value}' or
 * '{ubidEmailJSON.value:/regex/replacement/flags}' if a regular expression
 * is to be used.
 *
 * <h2>Escaping and Transforming Attribute Values</h2>
 *
 * To apply modifiers to the values extracted append ':' and then the name
 * of the modifier to apply. For example, if attribute 'mail' is to be
 * included in a constructed JSON value then modifier 'jsonEscape' could be
 * specified resulting in '{{ "userMail":"{mail:jsonEscape}" }}' or
 * '{{ "userMail":"{mail:/regex/replacement/flags:jsonEscape}" }}' if a
 * regular expression is to be used. Note that '{{' expands to '{' and
 * '}}' to '}'.
 *
 * <p>
 * The modifiers are:
 * <ul>
 *   <li>
 *     'dnEscape' : Escape text for use in a DN value.
 *   </li>
 *   <li>
 *     'jsonEscape' : Escape text for use in a JSON value.
 *   </li>
 *   <li>
 *     'ldapFilterEscape' : Escape text for use in an LDAP filter.
 *   </li>
 *   <li>
 *     'lowerCase' : Convert text to lower case.
 *   </li>
 *   <li>
 *     'trim' : Remove leading and trailing whitespace.
 *   </li>
 *   <li>
 *     'upperCase' : Convert text to upper case.
 *   </li>
 * </ul>
 *
 * <h2>Example</h2>
 * Here is an example to summarize these points.  Suppose, the 'uid'
 * field of the target entry needs to be populated with the initials of
 * the first and last name followed by the user's employee number which
 * is stored in the 'eid' attribute.  For example, 'John Smith' with an
 * employee id of 12345, would get a uid of 'js12345'.  The value-pattern
 * would be
 * '{givenname:/^(.)(.*)/$1/s:lowerCase}{sn:/^(.)(.*)/$1/s:lowerCase}{eid}'.
 * The employee's initials are extracted from the 'givenname' and 'sn'
 * attributes.  The '^(.)(.*)' regular expression used for these attributes
 * matches the entire value and stores the initial character in the $1
 * variable, which is used for the replacement.  (The 's' flag is used to
 * guard against the unlikely event that the user's 'givenname' or 'sn'
 * attribute includes a newline character.) The lowerCase modifier converts
 * the initial to lowercase.
 */
@ThreadSafety(level= ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface ValueConstructor
{
  /**
   * Returns a list of values constructed using the value template that was
   * used to create this ValueConstructor substituting attribute values from
   * the provided entry where they appear in the value template.
   *
   * @param  entry  Attributes referenced in the value template are pulled from
   *                this entry.
   *
   * @return  The list of attribute values based on the value template and the
   *          specified entry.
   *
   * @throws  LDAPException  If the provided entry does not include values for
   *                         all attributes that appear in the value template;
   *                         if the value template includes regular expressions
   *                         for extracting portions of an attribute value, and
   *                         the attribute value does not match the regular
   *                         expression; or if the value template references
   *                         multiple attributes that have more than one value.
   */
  List<String> constructValues(com.unboundid.ldap.sdk.Entry entry)
          throws LDAPException;


  /**
   * Returns a list of values constructed using the value template that was
   * used to create this ValueConstructor substituting attribute values from
   * the provided entry where they appear in the value template.
   *
   * @param  entry  Attributes referenced in the value template are pulled from
   *                this entry.
   *
   * @return  The list of attribute values based on the value template and the
   *          specified entry.
   *
   * @throws  LDAPException  If the provided entry does not include values for
   *                         all attributes that appear in the value template;
   *                         if the value template includes regular expressions
   *                         for extracting portions of an attribute value, and
   *                         the attribute value does not match the regular
   *                         expression; or if the value template references
   *                         multiple attributes that have more than one value.
   */
  List<String> constructValues(
          com.unboundid.directory.sdk.common.types.Entry entry)
          throws LDAPException;
}
