/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.api;



import com.unboundid.directory.sdk.common.operation.AddRequest;
import com.unboundid.directory.sdk.common.operation.AddResult;
import com.unboundid.directory.sdk.common.operation.DeleteRequest;
import com.unboundid.directory.sdk.common.operation.DeleteResult;
import com.unboundid.directory.sdk.common.operation.ModifyRequest;
import com.unboundid.directory.sdk.common.operation.ModifyResult;
import com.unboundid.directory.sdk.common.operation.ModifyDNRequest;
import com.unboundid.directory.sdk.common.operation.ModifyDNResult;
import com.unboundid.directory.sdk.common.types.CompletedOperationContext;
import com.unboundid.directory.sdk.common.types.Entry;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be implemented by an
 * extension which should be used to receive information about changes processed
 * within the server as the result of add, delete, modify, or modify DN
 * operations.  The {@code ServerContext.registerChangeListener} method should
 * be used to register a change listener with the server, and the corresponding
 * {@code deregisterChangeListener} method should be used to deregister the
 * listener if it is no longer needed.
 * <BR><BR>
 * Note that notification will only be provided for changes successfully
 * processed within the local server.  Notification will not be provided for
 * changes to data located in remote servers even if the change was processed
 * through this server.  Notification will also not be provided for operations
 * that did not complete successfully.
 * <BR><BR>
 * Also note that notification about matching changes will be provided to change
 * listeners asynchronously after the change has been processed within the
 * server and the result has been returned to the client.  Other changes to the
 * target entry may be in progress (and perhaps may have already completed) as
 * a result of other operations by the same or other clients.
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=false)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface ChangeListener
{
  /**
   * Indicates that an add operation has been processed within the server.
   *
   * @param  operationContext  The context for the add operation.
   * @param  addRequest        Information about the request for the add
   *                           operation that was processed.
   * @param  addResult         Information about the result for the add
   *                           operation that was processed.
   * @param  entry             The entry that was added to the server.
   */
  void addOperationProcessed(final CompletedOperationContext operationContext,
                             final AddRequest addRequest,
                             final AddResult addResult,
                             final Entry entry);



  /**
   * Indicates that a delete operation has been processed within the server.
   *
   * @param  operationContext  The context for the delete operation.
   * @param  deleteRequest     Information about the request for the delete
   *                           operation that was processed.
   * @param  deleteResult      Information about the result for the delete
   *                           operation that was processed.
   * @param  entry             The entry that was removed from the server.
   */
  void deleteOperationProcessed(
            final CompletedOperationContext operationContext,
            final DeleteRequest deleteRequest, final DeleteResult deleteResult,
            final Entry entry);



  /**
   * Indicates that a modify operation has been processed within the server.
   *
   * @param  operationContext  The context for the modify operation.
   * @param  modifyRequest     Information about the request for the modify
   *                           operation that was processed.
   * @param  modifyResult      Information about the result for the modify
   *                           operation that was processed.
   * @param  oldEntry          The entry as it appeared before the change was
   *                           processed.
   * @param  newEntry          The entry as it appeared immediately after the
   *                           change was processed.
   */
  void modifyOperationProcessed(
            final CompletedOperationContext operationContext,
            final ModifyRequest modifyRequest, final ModifyResult modifyResult,
            final Entry oldEntry, final Entry newEntry);



  /**
   * Indicates that a modify DN operation has been processed within the server.
   *
   * @param  operationContext  The context for the modify DN operation.
   * @param  modifyDNRequest   Information about the request for the modify DN
   *                           operation that was processed.
   * @param  modifyDNResult    Information about the result for the modify DN
   *                           operation that was processed.
   * @param  oldEntry          The entry as it appeared before the change was
   *                           processed.
   * @param  newEntry          The entry as it appeared immediately after the
   *                           change was processed.
   */
  void modifyDNOperationProcessed(
            final CompletedOperationContext operationContext,
            final ModifyDNRequest modifyDNRequest,
            final ModifyDNResult modifyDNResult, final Entry oldEntry,
            final Entry newEntry);
}
