/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.api;



import java.math.BigInteger;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.operation.AddRequest;
import com.unboundid.directory.sdk.common.operation.AddResult;
import com.unboundid.directory.sdk.common.operation.DeleteRequest;
import com.unboundid.directory.sdk.common.operation.DeleteResult;
import com.unboundid.directory.sdk.common.operation.ModifyRequest;
import com.unboundid.directory.sdk.common.operation.ModifyResult;
import com.unboundid.directory.sdk.common.operation.ModifyDNRequest;
import com.unboundid.directory.sdk.common.operation.ModifyDNResult;
import com.unboundid.directory.sdk.common.types.CompletedOperationContext;
import com.unboundid.directory.sdk.common.types.Entry;
import com.unboundid.directory.sdk.ds.config.ChangeSubscriptionHandlerConfig;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.ds.scripting.
            ScriptedChangeSubscriptionHandler;
import com.unboundid.directory.sdk.ds.types.ChangeSubscription;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which
 * receive notification of changes of interest processed within the server.
 * The server may be configured with one or more change subscriptions, which use
 * criteria to identify the types of changes that should be made available to
 * change subscription handlers.  Each change subscription handler may be
 * configured either to be notified only for changes matching a specific set of
 * change subscriptions, or for changes matching the criteria for any
 * subscription defined in the server.  This is handled automatically by the
 * server, so individual change subscription handler implementations do not need
 * to attempt to perform that filtering on their own.  However, they may perform
 * additional processing if desired to further narrow the set of changes to be
 * processed.
 * <BR>
 * <H2>Configuring Change Subscription Handlers</H2>
 * In order to configure a change subscription handler created using this API,
 * use a command like:
 * <PRE>
 *      dsconfig create-change-subscription-handler \
 *           --handler-name "<I>{handler-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{handler-name}</I>" is the name to use for the change subscription
 * handler instance, "<I>{class-name}</I>" is the fully-qualified name of the
 * Java class that extends
 * {@code com.unboundid.directory.sdk.ds.api.ChangeSubscriptionHandler}, and
 * "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the change subscription handler.  If multiple arguments should be
 * provided to the change subscription handler, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 *
 * @see  ScriptedChangeSubscriptionHandler
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=false)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class ChangeSubscriptionHandler
       implements UnboundIDExtension,
                  Reconfigurable<ChangeSubscriptionHandlerConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this change subscription handler.  All change
   * subscription handler implementations must include a default constructor,
   * but any initialization should generally be done in the
   * {@code initializeChangeSubscriptionHandler} method.
   */
  public ChangeSubscriptionHandler()
  {
    // No implementation is required.  However, we need to reference the
    // scripted change subscription handler so that checkstyle is satisfied
    // with the import.
    final ScriptedChangeSubscriptionHandler scriptedHandler = null;
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this change subscription handler.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this change
   *                        subscription handler.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this change subscription
   *                        handler.
   *
   * @throws  LDAPException  If a problem occurs while initializing this change
   *                         subscription handler.
   */
  public void initializeChangeSubscriptionHandler(
                   final DirectoryServerContext serverContext,
                   final ChangeSubscriptionHandlerConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final ChangeSubscriptionHandlerConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(
                         final ChangeSubscriptionHandlerConfig config,
                         final ArgumentParser parser,
                         final List<String> adminActionsRequired,
                         final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this change " +
                "subscription handler is disabled and re-enabled or until " +
                "the server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this change subscription
   * handler is to be taken out of service.
   */
  public void finalizeChangeSubscriptionHandler()
  {
    // No implementation is required.
  }



  /**
   * Performs any processing necessary for an add operation matching the
   * subscription criteria.
   *
   * @param  operationContext     The context for the add operation.
   * @param  sequenceNumber       The sequence number for the change
   *                              subscription notification.
   * @param  changeSubscriptions  The set of change subscriptions whose criteria
   *                              matched the add operation.
   * @param  addRequest           Information about the request for the add
   *                              operation that was processed.
   * @param  addResult            Information about the result for the add
   *                              operation that was processed.
   * @param  entry                The entry that was added to the server.
   */
  public abstract void addOperationProcessed(
                            final CompletedOperationContext operationContext,
                            final BigInteger sequenceNumber,
                            final Set<ChangeSubscription> changeSubscriptions,
                            final AddRequest addRequest,
                            final AddResult addResult, final Entry entry);



  /**
   * Performs any processing necessary for a delete operation matching the
   * subscription criteria.
   *
   * @param  operationContext     The context for the delete operation.
   * @param  sequenceNumber       The sequence number for the change
   *                              subscription notification.
   * @param  changeSubscriptions  The set of change subscriptions whose criteria
   *                              matched the delete operation.
   * @param  deleteRequest        Information about the request for the delete
   *                              operation that was processed.
   * @param  deleteResult         Information about the result for the delete
   *                              operation that was processed.
   * @param  entry                The entry that was removed from the server.
   */
  public abstract void deleteOperationProcessed(
                            final CompletedOperationContext operationContext,
                            final BigInteger sequenceNumber,
                            final Set<ChangeSubscription> changeSubscriptions,
                            final DeleteRequest deleteRequest,
                            final DeleteResult deleteResult,
                            final Entry entry);



  /**
   * Performs any processing necessary for a modify operation matching the
   * subscription criteria.
   *
   * @param  operationContext     The context for the modify operation.
   * @param  sequenceNumber       The sequence number for the change
   *                              subscription notification.
   * @param  changeSubscriptions  The set of change subscriptions whose criteria
   *                              matched the modify operation.
   * @param  modifyRequest        Information about the request for the modify
   *                              operation that was processed.
   * @param  modifyResult         Information about the result for the modify
   *                              operation that was processed.
   * @param  oldEntry             The entry as it appeared before the changes
   *                              were applied.
   * @param  newEntry             The entry as it appeared immediately after the
   *                              changes were applied.
   */
  public abstract void modifyOperationProcessed(
                            final CompletedOperationContext operationContext,
                            final BigInteger sequenceNumber,
                            final Set<ChangeSubscription> changeSubscriptions,
                            final ModifyRequest modifyRequest,
                            final ModifyResult modifyResult,
                            final Entry oldEntry, final Entry newEntry);



  /**
   * Performs any processing necessary for a modify DN operation matching the
   * subscription criteria.
   *
   * @param  operationContext     The context for the modify DN operation.
   * @param  sequenceNumber       The sequence number for the change
   *                              subscription notification.
   * @param  changeSubscriptions  The set of change subscriptions whose criteria
   *                              matched the modify DN operation.
   * @param  modifyDNRequest      Information about the request for the modify
   *                              DN operation that was processed.
   * @param  modifyDNResult       Information about the result for the modify DN
   *                              operation that was processed.
   * @param  oldEntry             The entry as it appeared before being renamed.
   * @param  newEntry             The entry as it appeared immediately after
   *                              being renamed.
   */
  public abstract void modifyDNOperationProcessed(
                            final CompletedOperationContext operationContext,
                            final BigInteger sequenceNumber,
                            final Set<ChangeSubscription> changeSubscriptions,
                            final ModifyDNRequest modifyDNRequest,
                            final ModifyDNResult modifyDNResult,
                            final Entry oldEntry, final Entry newEntry);



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
