/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2011-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.api;



import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.crypto.CipherInputStream;
import javax.crypto.CipherOutputStream;

import com.unboundid.directory.sdk.broker.internal.BrokerExtension;
import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.ds.config.CipherStreamProviderConfig;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which
 * provide access to cipher input streams and cipher output streams to be used
 * by the server in order to read and write encrypted data.
 * <BR>
 * <H2>Configuring Cipher Stream Providers</H2>
 * In order to configure a cipher stream provider created using this API, use a
 * command like:
 * <PRE>
 *      dsconfig create-cipher-stream-provider \
 *           --provider-name "<I>{provider-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{provider-name}</I>" is the name to use for the cipher stream
 * provider instance, "<I>{class-name}</I>" is the fully-qualified name of the
 * Java class that extends
 * {@code com.unboundid.directory.sdk.ds.api.CipherStreamProvider},
 * and "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the cipher stream provider.  If multiple arguments should be
 * provided to the cipher stream provider, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 */
@Extensible()
@DirectoryServerExtension()
@BrokerExtension()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class CipherStreamProvider
       implements UnboundIDExtension,
                  Reconfigurable<CipherStreamProviderConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this cipher stream provider.  All cipher stream
   * provider implementations must include a default constructor, but any
   * initialization should generally be done in the
   * {@code initializeCipherStreamProvider} method.
   */
  public CipherStreamProvider()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this cipher stream provider.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this cipher stream
   *                        provider.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this cipher stream provider.
   *
   * @throws  LDAPException  If a problem occurs while initializing this cipher
   *                         stream provider.
   */
  public void initializeCipherStreamProvider(
                   final DirectoryServerContext serverContext,
                   final CipherStreamProviderConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final CipherStreamProviderConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final CipherStreamProviderConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this cipher stream " +
                "provider is disabled and re-enabled or until the server is " +
                "restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this cipher stream
   * provider is to be taken out of service.
   */
  public void finalizeCipherStreamProvider()
  {
    // No implementation is required.
  }



  /**
   * Wraps the provided input stream in a cipher input stream that can be used
   * to decrypt data read from the given stream.
   *
   * @param  source  The input stream to be wrapped with a cipher input stream.
   *
   * @return  The cipher input stream which wraps the provided input stream.
   *
   * @throws  LDAPException  If a problem occurs while creating the cipher input
   *                         stream.
   */
  public abstract CipherInputStream createCipherInputStream(
                                         final InputStream source)
         throws LDAPException;



  /**
   * Wraps the provided output stream in a cipher output stream that can be used
   * to encrypt data written to the given stream.
   *
   * @param  target  The output stream to be wrapped with a cipher output
   *                 stream.
   *
   * @return  The cipher output stream which wraps the provided output stream.
   *
   * @throws  LDAPException  If a problem occurs while creating the cipher
   *                         output stream.
   */
  public abstract CipherOutputStream createCipherOutputStream(
                                          final OutputStream target)
         throws LDAPException;



  /**
   * Retrieves a map with information about any backup compatibility properties
   * that are specific to this cipher stream provider.
   *
   * @return  A map with information about any backup compatibility properties
   *          that are specific to this cipher stream provider.  It may be
   *          {@code null} or empty if there are no provider-specific
   *          properties.
   */
  public Map<String,String> getBackupCompatibilityProperties()
  {
    // No properties will be added by default.
    return null;
  }



  /**
   * Examines the provided set of backup compatibility properties to determine
   * whether there might be any warning or error conditions that may interfere
   * with the ability to restore a backup of the encryption settings database.
   * The default implementation does not do anything, but subclasses may
   * override this method to provide any appropriate warning and error messages.
   *
   * @param  sourceProperties  A map of properties (obtained from the
   *                           {@link #getBackupCompatibilityProperties} method)
   *                           from the backup to be restore.  The contents of
   *                           this map must not be altered.
   * @param  targetProperties  A map of properties (obtained from the
   *                           {@link #getBackupCompatibilityProperties} method)
   *                           from the server in which the backup is to be
   *                           restored.  The contents of this map must not be
   *                           altered.
   * @param  errorMessages     A list that may be updated with messages about
   *                           any compatibility errors that have been
   *                           identified.  If any compatibility errors are
   *                           identified, the restore will be aborted.
   * @param  warningMessages   A list that may be updated with messages about
   *                           any compatibility warnings that have been
   *                           identified.  If any compatibility warnings are
   *                           identified, they will be presented to a user
   *                           attempting a restore, but the user may choose to
   *                           ignore them if they are certain that the issue
   *                           will not cause any problems.
   */
  public void identifyBackupCompatibilityProblems(
                   final Map<String,String> sourceProperties,
                   final Map<String,String> targetProperties,
                   final List<CharSequence> errorMessages,
                   final List<CharSequence> warningMessages)
  {
    // No processing will be performed by default.
  }



  /**
   * Retrieves a list of human-readable string representations for the
   * provided cipher-stream-provider-specific backup compatibility properties.
   * The list returned does not need to have a one-to-one mapping with the
   * properties (e.g., it may omit information about some properties, or it may
   * combine information from multiple properties into a single string, or it
   * may convert one property into multiple strings).  It may also be
   * {@code null} if this cipher stream provider does not expect to have any
   * provider-specific properties.
   *
   * @param  propertyMap  A map of the property names and the associated values
   *                      to use in obtaining the human-readable string
   *                      representations.
   *
   * @return  A list of human-readable string representations for the
   *          provider-specific backup compatibility properties, or {@code null}
   *          if this cipher stream provider does not expect to have any
   *          provider-specific properties.
   */
  public List<String> getBackupCompatibilityPropertyStrings(
                           final Map<String,String> propertyMap)
  {
    // No property strings will be returned by default.
    return null;
  }



  /**
   * Retrieves a list of messages that should be logged (and made available in
   * the output) when backing up an encryption settings database protected with
   * this cipher stream provider.  For example, this may be used to warn about
   * additional files that may also need to be backed up separately (e.g., an
   * encryption key stored in a separate file).
   *
   * @return  A list of messages that should be logged when backing up an
   *          encryption settings database protected with this cipher stream
   *          provider.  It may be {@code null} or empty if no log messages are
   *          needed.
   */
  public List<String> getBackupLogMessages()
  {
    // No messages will be logged by default.
    return null;
  }



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
