/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2013-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.types.Entry;
import com.unboundid.directory.sdk.ds.config.PasswordStorageSchemeConfig;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.Modification;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.ByteString;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that may be implemented by extensions which provide
 * enhanced password storage scheme support.  Although the older
 * {@link PasswordStorageScheme} API is still supported, this enhanced version
 * is both simpler and provides additional functionality, including providing
 * access to the user entry (which may be useful, for example, if salt or
 * encryption information is to be derived from other content in the entry, or
 * from another repository that can be accessed based on content in the entry).
 * However, this API is generally simpler than the {@code PasswordStorageScheme}
 * API, so it may be desirable for all new implementations to implement this API
 * instead.
 * <BR><BR>
 * Encoded passwords may take one of two forms.  The first is the "user
 * password" syntax, in which the encoded password is represented by the name of
 * the storage scheme in curly braces followed by the transformed password
 * (e.g., "{scheme}encoded").  This format isn't based on any defined standard,
 * but is commonly used by a number of directory server implementations.  The
 * second format is the authentication password syntax as described in RFC 3112,
 * in which the encoded representation is broken into scheme, authInfo, and
 * authValue segments separated by dollar signs (e.g.,
 * "scheme$authInfo$authValue").  All password storage schemes are required to
 * support the "user password" syntax and may optionally also support the
 * authentication password syntax.
 * <BR>
 * <H2>Configuring Enhanced Password Storage Schemes</H2>
 * In order to configure a password storage scheme created using this API, use
 * a command like:
 * <PRE>
 *      dsconfig create-password-storage-scheme \
 *           --scheme-name "<I>{scheme-name}</I>" \
 *           --type third-party-enhanced \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{scheme-name}</I>" is the name to use for the password storage
 * scheme instance, "<I>{class-name}</I>" is the fully-qualified name of the
 * Java class that extends
 * {@code com.unboundid.directory.sdk.ds.api.PasswordStorageScheme}, and
 * "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the password storage scheme.  If multiple arguments should be
 * provided to the password storage scheme, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 *
 * @see PasswordStorageScheme
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=false)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class EnhancedPasswordStorageScheme
       implements UnboundIDExtension,
                  Reconfigurable<PasswordStorageSchemeConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this password storage scheme.  All password
   * storage scheme implementations must include a default constructor, but any
   * initialization should generally be done in the
   * {@code initializePasswordStorageScheme} method.
   */
  public EnhancedPasswordStorageScheme()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this password storage scheme.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this password storage
   *                        scheme.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this password storage scheme.
   *
   * @throws  LDAPException  If a problem occurs while initializing this
   *                         password storage scheme.
   */
  public void initializePasswordStorageScheme(
                   final DirectoryServerContext serverContext,
                   final PasswordStorageSchemeConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final PasswordStorageSchemeConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final PasswordStorageSchemeConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this password " +
                "storage scheme is disabled and re-enabled or until the " +
                "server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this password storage
   * scheme is to be taken out of service.
   */
  public void finalizePasswordStorageScheme()
  {
    // No implementation is required.
  }



  /**
   * Retrieves the name for this password storage scheme.  This will be the
   * identifier which appears in curly braces at the beginning of the encoded
   * password.  The name should not include curly braces.
   *
   * @return  The name for this password storage scheme.
   */
  public abstract String getStorageSchemeName();



  /**
   * Indicates whether this password storage scheme encodes passwords in a form
   * that allows the original plaintext value to be obtained from the encoded
   * representation.
   *
   * @return  {@code true} if the original plaintext password may be obtained
   *          from the encoded password, or {@code false} if not.
   */
  public abstract boolean isReversible();



  /**
   * Indicates whether this password storage scheme encodes passwords in a form
   * that may be considered secure.  A storage scheme should only be considered
   * secure if it is not possible to trivially determine a clear-text value
   * which may be used to generate a given encoded representation.
   *
   * @return  {@code true} if this password storage scheme may be considered
   *          secure, or {@code false} if not.
   */
  public abstract boolean isSecure();



  /**
   * Indicates whether this password storage scheme requires access to the user
   * entry in order to perform processing.  If the storage scheme does require
   * access to the user entry (e.g., in order to obtain key or salt information
   * from another attribute, or to use information in the entry to access
   * information in another data store), then some features (e.g., the ability
   * to perform extensible matching with password values, or the ability to use
   * the encode-password tool to encode and compare passwords) may not be
   * available.
   * <BR><BR>
   * Note that if this method returns {@code true}, then the server will never
   * provide a {@code null} value for the {@code userEntry} parameter to methods
   * that accept it.  However, if this method returns {@code false}, then there
   * may be some instances in which the server may invoke those methods with a
   * {@code null} value for the {@code userEntry} parameter (e.g., when invoked
   * by the encode-password tool or when targeting a password with an extensible
   * match search filter).
   *
   * @return  {@code true} if this password storage scheme requires access to
   *          the user entry, or {@code false} if not.
   */
  public boolean requiresUserEntry()
  {
    return false;
  }



  /**
   * Encodes the provided plaintext password for this storage scheme,
   * without the name of the associated scheme.  Note that the
   * provided plaintext password should not be altered in any way.
   *
   * @param  plaintext      The plaintext password to be encoded.  It must not
   *                        be {@code null}.  Note that there is no guarantee
   *                        that password validators have yet been invoked for
   *                        this password, so this password storage scheme
   *                        implementation should not make any assumptions about
   *                        the format of the plaintext password or whether it
   *                        will actually be allowed for use in the entry.
   * @param  userEntry      The complete entry for the user for whom the
   *                        password is to be encoded.  This will not be
   *                        {@code null} for schemes in which
   *                        {@link #requiresUserEntry} returns {@code true}.
   * @param  modifications  The set of modifications to be applied to the user
   *                        entry.  This will generally be non-{@code null} only
   *                        for operations that use a modify to change the user
   *                        password.  If a modification list is provided, then
   *                        it will contain the complete set of modifications
   *                        for the operation, some of which may have no
   *                        impact on password encoding.
   * @param  deterministic  Indicates whether the password encoding should be
   *                        deterministic.  If this is {@code true}, then the
   *                        scheme should attempt to generate a consistent
   *                        encoding for the password (e.g., by determining the
   *                        salt from a normalized representation of the user's
   *                        DN).  This will be used during LDIF import
   *                        processing (and potentially at other times) to help
   *                        ensure that if the LDIF file is imported into
   *                        multiple servers, any clear-text password
   *                        encountered in the LDIF file will have the same
   *                        encoded representation on all servers.  Some
   *                        password storage schemes may choose to ignore this
   *                        property if it does not apply or is not possible to
   *                        achieve.
   * @param  includeScheme  Indicates whether to include the name of the scheme
   *                        in curly braces before the encoded password.
   *
   * @return  The password that has been encoded using this storage
   *          scheme.
   *
   * @throws  LDAPException  If a problem occurs while attempting to encode the
   *                         password.
   */
  public abstract ByteString encodePassword(final ByteString plaintext,
                                  final Entry userEntry,
                                  final List<Modification> modifications,
                                  final boolean deterministic,
                                  final boolean includeScheme)
         throws LDAPException;



  /**
   * Indicates whether the provided plaintext password could have been used to
   * generate the given encoded password.
   *
   * @param  plaintextPassword  The plaintext password provided by the user as
   *                            part of a simple bind attempt.
   * @param  storedPassword     The stored password to compare against the
   *                            provided plaintext password.  It will not
   *                            include the scheme name in curly braces.
   * @param  userEntry          The complete entry for the user for whom the
   *                            password is to be validated.  This will not be
   *                            {@code null} for schemes in which
   *                            {@link #requiresUserEntry} returns {@code true}.
   *
   * @return  {@code true} if the provided clear-text password could have been
   *          used to generate the encoded password, or {@code false} if not.
   */
  public abstract boolean passwordMatches(final ByteString plaintextPassword,
                                          final ByteString storedPassword,
                                          final Entry userEntry);



  /**
   * Attempts to determine the plaintext password used to generate the provided
   * encoded password.  This method should only be called if the
   * {@link #isReversible} method returns {@code true}.
   *
   * @param  storedPassword  The password for which to obtain the plaintext
   *                         value.  It should not include the scheme name in
   *                         curly braces.
   * @param  userEntry       The complete entry for the user for whom the
   *                         password is to be validated.  This will not be
   *                         {@code null} for schemes in which
   *                         {@link #requiresUserEntry} returns {@code true}.
   *
   * @return  The plaintext password obtained from the given encoded password.
   *
   * @throws  LDAPException  If this password storage scheme is not reversible,
   *                         or if the provided value could not be decoded to
   *                         its plaintext representation.
   */
  public abstract ByteString getPlaintextValue(final ByteString storedPassword,
                                               final Entry userEntry)
         throws LDAPException;



  /**
   * Indicates whether this password storage scheme provides the ability to
   * encode passwords in the authentication password syntax as described in RFC
   * 3112.
   *
   * @return  {@code true} if this password storage scheme supports the
   *          authentication password syntax, or {@code false} if not.
   */
  public boolean supportsAuthPasswordSyntax()
  {
    return false;
  }



  /**
   * Retrieves the name that should be used to identify this password storage
   * scheme when encoding passwords using the authentication password syntax as
   * described in RFC 3112.  This should only be used if the
   * {@link #supportsAuthPasswordSyntax} method returns {@code true}.
   *
   * @return  The name that should be used to identify this password storage
   *          scheme when encoding passwords using the authentication password
   *          syntax.
   */
  public String getAuthPasswordSchemeName()
  {
    return getStorageSchemeName();
  }



  /**
   * Encodes the provided plaintext password using the authentication password
   * syntax as defined in RFC 3112.  This should only be used if the
   * {@link #supportsAuthPasswordSyntax} method returns {@code true}.
   *
   * @param  plaintext      The plaintext password to be encoded.  It must not
   *                        be {@code null}.  Note that there is no guarantee
   *                        that password validators have yet been invoked for
   *                        this password, so this password storage scheme
   *                        implementation should not make any assumptions about
   *                        the format of the plaintext password or whether it
   *                        will actually be allowed for use in the entry.
   * @param  userEntry      The complete entry for the user for whom the
   *                        password is to be encoded.  This will not be
   *                        {@code null} for schemes in which
   *                        {@link #requiresUserEntry} returns {@code true}.
   * @param  modifications  The set of modifications to be applied to the user
   *                        entry.  This will generally be non-{@code null} only
   *                        for operations that use a modify to change the user
   *                        password.  If a modification list is provided, then
   *                        it will contain the complete set of modifications
   *                        for the operation, some of which may have no
   *                        impact on password encoding.
   * @param  deterministic  Indicates whether the password encoding should be
   *                        deterministic.  If this is {@code true}, then the
   *                        scheme should attempt to generate a consistent
   *                        encoding for the password (e.g., by determining the
   *                        salt from a normalized representation of the user's
   *                        DN).  This will be used during LDIF import
   *                        processing (and potentially at other times) to help
   *                        ensure that if the LDIF file is imported into
   *                        multiple servers, any clear-text password
   *                        encountered in the LDIF file will have the same
   *                        encoded representation on all servers.  Some
   *                        password storage schemes may choose to ignore this
   *                        property if it does not apply or is not possible to
   *                        achieve.
   *
   * @return  The encoded representation of the provided password.
   *
   * @throws  LDAPException  If a problem occurs while encoding the provided
   *                         password, or if this password storage scheme does
   *                         not support the authentication password syntax.
   */
  public ByteString encodeAuthPassword(final ByteString plaintext,
                                       final Entry userEntry,
                                       final List<Modification> modifications,
                                       final boolean deterministic)
         throws LDAPException
  {
    throw new LDAPException(ResultCode.INVALID_ATTRIBUTE_SYNTAX,
         "This password storage scheme does not support the use of the " +
         "authentication password syntax.");
  }



  /**
   * Indicates whether the provided plaintext password may be used to generate
   * an encoded password with the given authInfo and authValue elements when
   * using the authentication password syntax as defined in RFC 3112.  This
   * should only be used if the {@link #supportsAuthPasswordSyntax} method
   * returns {@code true}.
   *
   * @param  plaintextPassword  The plaintext password provided by the user.
   * @param  authInfo           The authInfo component of the password encoded
   *                            in the authentication password syntax.
   * @param  authValue          The authValue component of the password encoded
   *                            in the authentication password syntax.
   * @param  userEntry          The complete entry for the user for whom the
   *                            password is to be validated.  This will not be
   *                            {@code null} for schemes in which
   *                            {@link #requiresUserEntry} returns {@code true}.
   *
   * @return  {@code true} if the provided plaintext password could be used to
   *          generate an encoded password with the given authInfo and authValue
   *          portions, or {@code false} if not.
   */
  public boolean authPasswordMatches(final ByteString plaintextPassword,
                                     final String authInfo,
                                     final String authValue,
                                     final Entry userEntry)
  {
    return false;
  }



  /**
   * Obtains the plaintext password that was used to generate an encoded
   * password with the given authInfo and authValue elements when using the
   * authentication password syntax as described in RFC 3112.  This should only
   * be used if both the {@link #supportsAuthPasswordSyntax} and
   * {@link #isReversible} methods return {@code true}.
   *
   * @param  authInfo   The authInfo component of the password encoded in the
   *                    authentication password syntax.
   * @param  authValue  The authValue component of the password encoded in the
   *                    authentication password syntax.
   * @param  userEntry  The complete entry for the user for whom the password is
   *                    to be validated.  This will not be {@code null} for
   *                    schemes in which {@link #requiresUserEntry} returns
   *                    {@code true}.  The entry should not be altered in any
   *                    way by this password storage scheme.
   *
   * @return  The plaintext password that was used to generate the encoded
   *          password.
   *
   * @throws  LDAPException  If this password storage scheme is not reversible,
   *                         if this password storage scheme does not support
   *                         the authentication password syntax, or if some
   *                         other problem is encountered while attempting to
   *                         determine the plaintext password.
   */
  public ByteString getAuthPasswordPlaintextValue(final String authInfo,
                                                  final String authValue,
                                                  final Entry userEntry)
         throws LDAPException
  {
    throw new LDAPException(ResultCode.INVALID_ATTRIBUTE_SYNTAX,
         "This password storage scheme does not support the use of the " +
         "authentication password syntax.");
  }



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
