/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.operation.ExtendedRequest;
import com.unboundid.directory.sdk.common.operation.UpdatableExtendedResult;
import com.unboundid.directory.sdk.common.types.OperationContext;
import com.unboundid.directory.sdk.ds.config.ExtendedOperationHandlerConfig;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which are
 * used to process extended operations within the server.  Extended operations
 * may be used to perform processing which falls outside of that available by
 * the core set of LDAP operations.  Each extended request will include a
 * numeric object identifier (OID) which indicates the type of request to be
 * processed, and may also include a value with additional information about the
 * processing to be performed.  Each extended operation handler must specify the
 * OIDs of the types of extended requests that it may be used to process.
 * <BR>
 * <H2>Configuring Extended Operation Handlers</H2>
 * In order to configure an extended operation handler created using this API,
 * use a command like:
 * <PRE>
 *      dsconfig create-extended-operation-handler \
 *           --handler-name "<I>{handler-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{handler-name}</I>" is the name to use for the extended operation
 * handler instance, "<I>{class-name}</I>" is the fully-qualified name of the
 * Java class that extends
 * {@code com.unboundid.directory.sdk.ds.api.ExtendedOperationHandler}, and
 * "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the extended operation handler.  If multiple arguments should be
 * provided to the extended operation handler, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=true,
     notes="Some extended operation handlers may only apply to local content")
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class ExtendedOperationHandler
       implements UnboundIDExtension,
                  Reconfigurable<ExtendedOperationHandlerConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this extended operation handler.  All extended
   * operation handler implementations must include a default constructor, but
   * any initialization should generally be done in the
   * {@code initializeExtendedOperationHandler} method.
   */
  public ExtendedOperationHandler()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this extended operation handler.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this extended
   *                        operation handler.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this extended operation
   *                        handler.
   *
   * @throws  LDAPException  If a problem occurs while initializing this
   *                         extended operation handler.
   */
  public void initializeExtendedOperationHandler(
                   final DirectoryServerContext serverContext,
                   final ExtendedOperationHandlerConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final ExtendedOperationHandlerConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(
                         final ExtendedOperationHandlerConfig config,
                         final ArgumentParser parser,
                         final List<String> adminActionsRequired,
                         final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this extended " +
                "operation handler is disabled and re-enabled or until the " +
                "server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this extended operation
   * handler is to be taken out of service.
   */
  public void finalizeExtendedOperationHandler()
  {
    // No implementation is required.
  }



  /**
   * Retrieves the name of the extended operation with the provided OID.
   *
   * @param  oid  The OID of the extended operation for which to retrieve the
   *              corresponding name.
   *
   * @return  The name of the extended operation with the specified OID, or
   *          {@code null} if the specified OID is not recognized by this
   *          extended operation handler.
   */
  public abstract String getExtendedOperationName(final String oid);



  /**
   * Retrieves the OIDs of the extended operation types supported by this
   * extended operation handler.
   *
   * @return  The OIDs of the extended operation types supported by this
   *          extended operation handler.  It must not be {@code null} or
   *          empty, and the contents of the set returned must not change over
   *          the life of this extended operation handler.
   */
  public abstract Set<String> getSupportedExtensions();



  /**
   * Retrieves the OIDs of any controls supported by this extended operation
   * handler.
   *
   * @return  The OIDs of any controls supported by this extended operation
   *          handler.  It may be {@code null} or empty if this extended
   *          operation handler does not support any controls.
   */
  public Set<String> getSupportedControls()
  {
    return Collections.emptySet();
  }



  /**
   * Retrieves the OIDs of any features supported by this extended operation
   * handler that should be advertised in the server root DSE.
   *
   * @return  The OIDs of any features supported by this extended operation
   *          handler.  It may be {@code null} or empty if this extended
   *          operation handler does not support any features.
   */
  public Set<String> getSupportedFeatures()
  {
    return Collections.emptySet();
  }



  /**
   * Performs any processing appropriate for the provided extended request.
   *
   * @param  operationContext  The operation context for the extended operation.
   * @param  request           The extended request to be processed.
   * @param  result            The extended result to be updated with the result
   *                           of processing.
   */
  public abstract void processExtendedOperation(
                            final OperationContext operationContext,
                            final ExtendedRequest request,
                            final UpdatableExtendedResult result);



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
