/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.ds.config.IdentityMapperConfig;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which
 * attempt to map a username to a user defined in the server.  This is generally
 * used when processing an authorization ID, as might be provided when
 * performing SASL authentication or in a control like the proxied authorization
 * or intermediate client controls.  In order for the mapping to be established,
 * the identity mapper must locate exactly one entry in the server corresponding
 * to the provided username.  If no entries are found, or if multiple entries
 * are found, then the mapping attempt must fail.
 * <BR>
 * <H2>Configuring Identity Mappers</H2>
 * In order to configure an identity mapper created using this API, use a
 * command like:
 * <PRE>
 *      dsconfig create-identity-mapper \
 *           --mapper-name "<I>{mapper-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{mapper-name}</I>" is the name to use for the identity mapper
 * instance, "<I>{class-name}</I>" is the fully-qualified name of the Java class
 * that extends {@code com.unboundid.directory.sdk.ds.api.IdentityMapper},
 * and "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the identity mapper.  If multiple arguments should be provided to
 * the identity mapper, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 *
 * @see  com.unboundid.directory.sdk.ds.scripting.ScriptedIdentityMapper
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=true,
     notes="Identity mappers may only apply in certain contexts for proxied " +
          "operations (e.g., when used in the context of SASL authentication " +
          "but not in the value of the proxied authorization control).")
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class IdentityMapper
       implements UnboundIDExtension, Reconfigurable<IdentityMapperConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this identity mapper.  All identity mapper
   * implementations must include a default constructor, but any initialization
   * should generally be done in the {@code initializeIdentityMapper} method.
   */
  public IdentityMapper()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this identity mapper.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this identity mapper.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this identity mapper.
   *
   * @throws  LDAPException  If a problem occurs while initializing this
   *                         identity mapper.
   */
  public void initializeIdentityMapper(
                   final DirectoryServerContext serverContext,
                   final IdentityMapperConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(final IdentityMapperConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final IdentityMapperConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this identity " +
                "mapper is disabled and re-enabled or until the server is " +
                "restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this identity mapper is to
   * be taken out of service.
   */
  public void finalizeIdentityMapper()
  {
    // No implementation is required.
  }



  /**
   * Performs any processing which may be necessary to map the provided username
   * to a user within the server.
   *
   * @param  username  The username to be mapped to a user within the server.
   *
   * @return  The DN of the user within the server to which the provided
   *          username corresponds.
   *
   * @throws  LDAPException  If the provided username cannot be mapped to
   *                         exactly one user in the server.
   */
  public abstract String mapUsername(final String username)
         throws LDAPException;



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
