/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2014-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.api;

import com.unboundid.asn1.ASN1OctetString;
import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.operation.AddRequest;
import com.unboundid.directory.sdk.common.operation.DeleteRequest;
import com.unboundid.directory.sdk.common.operation.ModifyDNRequest;
import com.unboundid.directory.sdk.common.operation.ModifyRequest;
import com.unboundid.directory.sdk.common.types.Entry;
import com.unboundid.directory.sdk.common.types.OperationContext;
import com.unboundid.directory.sdk.ds.config.NotificationManagerConfig;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.ds.types.Notification;
import com.unboundid.directory.sdk.ds.types.NotificationDeliveryResult;
import com.unboundid.directory.sdk.ds.types.NotificationProperties;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.ldap.sdk.unboundidds.extensions.
    NotificationDestinationDetails;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;

import java.util.Collections;
import java.util.List;
import java.util.Map;



/**
 * This class defines an API that must be implemented by extensions which
 * wish to deliver notification of changes of interest processed within the
 * server.
 * <BR>
 * <H2>Implementing a Notification Manager</H2>
 * <H3>Subscription management</H3>
 * A notification manager has one or more notification destinations which
 * determine where notifications are to be delivered. Each notification
 * destination has one or more notification subscriptions which determine
 * which changes should result in a notification for that destination.
 * Destinations and subscriptions each have associated details whose syntax
 * is specific to the notification manager implementation.
 * The notification manager implementation must implement the following
 * methods to keep track of destinations, subscriptions and their associated
 * details:
 * <UL>
 *   <LI>initializeNotificationDestinations</LI>
 *   <LI>areDestinationDetailsAcceptable</LI>
 *   <LI>setNotificationDestination</LI>
 *   <LI>deleteNotificationDestination</LI>
 *   <LI>areSubscriptionDetailsAcceptable</LI>
 *   <LI>setNotificationSubscription</LI>
 *   <LI>deleteNotificationSubscription</LI>
 * </UL>
 * <H3>Matching changes with subscriptions</H3>
 * A notification manager must implement the following methods (one for each
 * type of LDAP operation) to determine whether a change matches any
 * notification subscriptions. The notification manager may provide arbitrary
 * properties that it wishes to be included with a notification:
 * <UL>
 *   <LI>getAddNotificationProperties</LI>
 *   <LI>getDeleteNotificationProperties</LI>
 *   <LI>getModifyNotificationProperties</LI>
 *   <LI>getModifyDNNotificationProperties</LI>
 * </UL>
 * <H3>Delivering notifications</H3>
 * A notification manager must implement the following method to attempt
 * delivery of a notification to a destination:
 * <UL>
 *   <LI>attemptDelivery</LI>
 * </UL>
 * <BR>
 * <H2>Configuring a Notification Manager</H2>
 * The LDAP changelog must first be enabled on each server that is to deliver
 * notifications:
 * <PRE>
 *      dsconfig set-backend-prop \
 *           --backend-name changelog \
 *           --set enabled:true
 * </PRE>
 * In order to configure a notification manager created using this API,
 * use a command like:
 * <PRE>
 *      dsconfig create-notification-manager \
 *           --manager-name "<I>{manager-ID}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "subscription-base-dn:<I>{subscription-base-dn}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{manager-ID}</I>" is the ID to use for the notification
 * manager instance, "<I>{subscription-base-dn}</I>" is the base DN where
 * subscription information for this notification manager is to be stored,
 * "<I>{class-name}</I>" is the fully-qualified name of the Java class that
 * extends
 * {@code com.unboundid.directory.sdk.ds.api.NotificationManager}, and
 * "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the notification manager.  If multiple arguments should be
 * provided to the notification manager, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 * <BR>
 * <BR>
 * The notification manager must also be associated with a backend using a
 * command like:
 * <PRE>
 *      dsconfig set-backend-prop \
 *           --backend-name "<I>userRoot</I>" \
 *           --set "notification-manager:<I>{manager-ID}</I>"
 * </PRE>
 * The <I>{subscription-base-dn}</I> of the notification manager must be
 * within the scope of the <I>base-dn</I> of the backend.
 * <BR>
 * <BR>
 * The above configuration should be made on each server that holds a replica
 * of the backend data to enable delivery of notifications for changes to that
 * data.
 */
@Extensible()
@DirectoryServerExtension()
@ThreadSafety(level= ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class NotificationManager
    implements UnboundIDExtension,
               Reconfigurable<NotificationManagerConfig>,
               ExampleUsageProvider
{
  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this notification manager.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this notification
   *                        manager.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this notification manager.
   *
   * @throws  LDAPException  If a problem occurs while initializing this
   *                         notification manager.
   */
  public void initializeNotificationManager(
                   final DirectoryServerContext serverContext,
                   final NotificationManagerConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * Initializes or re-initializes the notification destinations for this
   * notification manager. This will be called once after the notification
   * manager has been initialized and any time subsequently.
   *
   * @param destinations  The list of defined notification destinations and
   *                      their associated subscriptions.
   *
   * @throws  LDAPException  If a problem occurs while initializing the
   *                         notification destinations for this notification
   *                         manager.
   */
  public abstract void initializeNotificationDestinations(
      List<NotificationDestinationDetails> destinations)
      throws LDAPException;



  /**
   * Determine whether the provided destination details are acceptable.
   * If this method returns true then it is expected that a call to
   * {@code setNotificationDestination} with the same details will not
   * fail due to invalid details.
   * @param destinationID        The notification destination ID.
   * @param destinationDetails   The notification destination details.
   * @param unacceptableReasons  A list that may be used to hold the
   *                             reasons that the provided details are
   *                             not acceptable.
   * @return {@code true} if the provided details are acceptable.
   */
  public abstract boolean areDestinationDetailsAcceptable(
      String destinationID,
      List<ASN1OctetString> destinationDetails,
      List<String> unacceptableReasons);


  /**
   * Create or update a notification destination.
   * @param destinationID       The notification destination ID.
   * @param destinationDetails  The notification destination details.
   * @throws  LDAPException  If a problem occurs while creating or updating the
   *                         notification destination.
   */
  public abstract void setNotificationDestination(
      String destinationID,
      List<ASN1OctetString> destinationDetails)
      throws LDAPException;



  /**
   * Determine whether the provided subscription details are acceptable.
   * If this method returns true then it is expected that a call to
   * setNotificationSubscription with the same details will not fail
   * due to invalid details.
   * @param destinationID        The notification destination ID.
   * @param subscriptionID       The notification subscription ID.
   * @param subscriptionDetails  The notification subscription details.
   * @param unacceptableReasons  A list that may be used to hold the
   *                             reasons that the provided details are
   *                             not acceptable.
   * @return {@code true} if the provided details are acceptable.
   */
  public abstract boolean areSubscriptionDetailsAcceptable(
      String destinationID,
      String subscriptionID,
      List<ASN1OctetString> subscriptionDetails,
      List<String> unacceptableReasons);



  /**
   * Create or update a notification subscription.
   * @param destinationID        The notification destination ID.
   * @param subscriptionID       The notification subscription ID.
   * @param subscriptionDetails  The notification destination details.
   * @throws  LDAPException  If a problem occurs while creating or updating the
   *                         notification subscription.
   */
  public abstract void setNotificationSubscription(
      String destinationID,
      String subscriptionID,
      List<ASN1OctetString> subscriptionDetails)
      throws LDAPException;



  /**
   * Delete a notification destination including any associated
   * subscriptions.
   * @param destinationID  The notification destination ID.
   */
  public abstract void deleteNotificationDestination(String destinationID);



  /**
   * Delete a notification subscription.
   * @param destinationID   The notification destination ID.
   * @param subscriptionID  The notification subscription ID.
   */
  public abstract void deleteNotificationSubscription(String destinationID,
                                                      String subscriptionID);



  /**
   * Determine whether the server running this extension is preferred for the
   * given notification destination. For example, this method could return
   * {@code true} if this server is local to the destination.
   *
   * @param destinationID  The notification destination ID.
   *
   * @return  {@code true} if this server is preferred for the given
   *          notification destination.
   */
  public abstract boolean isPreferredForDestination(String destinationID);



  /**
   * This method is only provided for backwards compatibility. Subclasses
   * should implement the alternate version of getAddNotificationProperties
   * below. In which case, this method will not be called.
   *
   * Determine if any notifications are required for the provided add
   * request and return notification properties for each notification
   * destination that requires a notification.
   *
   * @param addRequest  The add request that is being processed.
   *
   * @return   A list of notification properties with an element for each
   *           notification destination that requires a notification. An
   *           empty or {@code null} list indicates that the operation does not
   *           require any notifications.
   *
   * @deprecated  Override the other getAddNotificationProperties method
   *              instead.
   */
  @Deprecated
  public List<NotificationProperties> getAddNotificationProperties(
      final AddRequest addRequest)
  {
    throw new UnsupportedOperationException(
            "One of the getAddNotificationProperties() methods " +
            "must be implemented");
  }



  /**
   * Determine if any notifications are required for the provided add
   * request and return notification properties for each notification
   * destination that requires a notification.
   *
   * @param addRequest  The add request that is being processed.
   *
   * @param addedEntry  The entry that was added.
   *
   * @param opContext   The operation context for the current operation.
   *
   * @return   A list of notification properties with an element for each
   *           notification destination that requires a notification. An
   *           empty or {@code null} list indicates that the operation does not
   *           require any notifications.
   */
  @SuppressWarnings("deprecation")
  public List<NotificationProperties> getAddNotificationProperties(
      final AddRequest addRequest,
      final Entry addedEntry,
      final OperationContext opContext)
  {
    return getAddNotificationProperties(addRequest);
  }



  /**
   * This method is only provided for backwards compatibility. Subclasses
   * should implement the alternate version of getDeleteNotificationProperties
   * below. In which case, this method will not be called.
   *
   * Determine if any notifications are required for the provided delete
   * request and return notification properties for each notification
   * destination that requires a notification.
   *
   * @param deleteRequest  The delete request that is being processed.
   *
   * @return   A list of notification properties with an element for each
   *           notification destination that requires a notification. An
   *           empty or {@code null} list indicates that the operation does not
   *           require any notifications.
   *
   * @deprecated  Override the other getDeleteNotificationProperties method
   *              instead.
   */
  @Deprecated
  public List<NotificationProperties> getDeleteNotificationProperties(
      final DeleteRequest deleteRequest)
  {
    throw new UnsupportedOperationException(
            "One of the getDeleteNotificationProperties() methods " +
            "must be implemented");
  }



  /**
   * Determine if any notifications are required for the provided delete
   * request and return notification properties for each notification
   * destination that requires a notification.
   *
   * @param deleteRequest  The delete request that is being processed.
   *
   * @param deletedEntry   The entry against which the delete operation was
   *                       processed, if available, and {@code null} otherwise.
   *
   * @param opContext      The operation context for the current operation.
   *
   * @return   A list of notification properties with an element for each
   *           notification destination that requires a notification. An
   *           empty or {@code null} list indicates that the operation does not
   *           require any notifications.
   */
  @SuppressWarnings("deprecation")
  public List<NotificationProperties> getDeleteNotificationProperties(
      final DeleteRequest deleteRequest,
      final Entry deletedEntry,
      final OperationContext opContext)
  {
    return getDeleteNotificationProperties(deleteRequest);
  }



  /**
   * This method is only provided for backwards compatibility. Subclasses
   * should implement the alternate version of getModifyNotificationProperties
   * below. In which case, this method will not be called.
   *
   * Determine if any notifications are required for the provided modify
   * request and return notification properties for each notification
   * destination that requires a notification.
   *
   * @param modifyRequest  The modify request that is being processed.
   *
   * @return   A list of notification properties with an element for each
   *           notification destination that requires a notification. An
   *           empty or {@code null} list indicates that the operation does not
   *           require any notifications.
   *
   * @deprecated  Override the other getModifyNotificationProperties method
   *              instead.
   */
  @Deprecated
  public List<NotificationProperties> getModifyNotificationProperties(
      final ModifyRequest modifyRequest)
  {
    throw new UnsupportedOperationException(
            "One of the getModifyNotificationProperties() methods " +
            "must be implemented");
  }



  /**
   * Determine if any notifications are required for the provided modify
   * request and return notification properties for each notification
   * destination that requires a notification.
   *
   * @param modifyRequest  The modify request that is being processed.
   *
   * @param oldEntry       The entry as it appeared before the modify operation,
   *                       or {@code null} if it is not available.
   *
   * @param newEntry       The entry as it appeared after the modify operation,
   *                       or {@code null} if it is not available.
   *
   * @param opContext      The operation context for the current operation.
   *
   * @return   A list of notification properties with an element for each
   *           notification destination that requires a notification. An
   *           empty or {@code null} list indicates that the operation does not
   *           require any notifications.
   */
  @SuppressWarnings("deprecation")
  public List<NotificationProperties> getModifyNotificationProperties(
      final ModifyRequest modifyRequest,
      final Entry oldEntry,
      final Entry newEntry,
      final OperationContext opContext)
  {
    return getModifyNotificationProperties(modifyRequest);
  }



  /**
   * This method is only provided for backwards compatibility. Subclasses
   * should implement the alternate version of getModifyDNNotificationProperties
   * below. In which case, this method will not be called.
   *
   * Determine if any notifications are required for the provided modify DN
   * request and return notification properties for each notification
   * destination that requires a notification.
   *
   * @param modifyDNRequest  The modify DN request that is being processed.
   *
   * @return   A list of notification properties with an element for each
   *           notification destination that requires a notification. An
   *           empty or {@code null} list indicates that the operation does not
   *           require any notifications.
   *
   * @deprecated  Override the other getModifyDNNotificationProperties method
   *              instead.
   */
  @Deprecated
  public List<NotificationProperties>
  getModifyDNNotificationProperties(
      final ModifyDNRequest modifyDNRequest)
  {
    throw new UnsupportedOperationException(
            "One of the getModifyDNNotificationProperties() methods " +
            "must be implemented");
  }



  /**
   * Determine if any notifications are required for the provided modify DN
   * request and return notification properties for each notification
   * destination that requires a notification.
   *
   * @param modifyDNRequest  The modify DN request that is being processed.
   *
   * @param oldEntry         The entry as it appeared before the modify DN
   *                         operation, or {@code null} if it is not available.
   *
   * @param newEntry         The entry as it appeared after the modify DN
   *                         operation,  or {@code null} if it is not available.
   *
   * @param opContext        The operation context for the current operation.
   *
   * @return   A list of notification properties with an element for each
   *           notification destination that requires a notification. An
   *           empty or {@code null} list indicates that the operation does not
   *           require any notifications.
   */
  @SuppressWarnings("deprecation")
  public List<NotificationProperties>
  getModifyDNNotificationProperties(
      final ModifyDNRequest modifyDNRequest,
      final Entry oldEntry,
      final Entry newEntry,
      final OperationContext opContext)
  {
    return getModifyDNNotificationProperties(modifyDNRequest);
  }



  /**
   * Attempt delivery of a notification and return a result indicating whether
   * delivery was successful, and whether delivery should be retried if this
   * attempt was unsuccessful. Notification manager implementations should be
   * careful not to return {@code RETRY} when all future attempts of the
   * notification delivery will fail, e.g. a remote change failing due to a
   * schema violation. If the extension can determine that the remote service
   * is completely unavailable, then it is fine to continue to retry, but if
   * the service is available and only failing for some changes, then
   * continuing to retry is dangerous. There are methods on the {@code
   * Notification} interface to determine how many attempts have been made
   * and for how long attempts have been made. Above some threshold, the
   * extension should return {@code FAILURE} instead of {@code RETRY}.
   * <br>
   * <br>
   * This method must be written to be thread-safe because notifications of
   * changes that do not depend on each other are processed in parallel (e.g.
   * when the changes affect unrelated entries).
   *
   * @param notification  The notification to be delivered.
   *
   * @return  A delivery result indicating whether delivery was successful,
   *          and whether delivery should be retried if this attempt was
   *          unsuccessful.
   */
  public abstract NotificationDeliveryResult attemptDelivery(
      Notification notification);



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final NotificationManagerConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(
                         final NotificationManagerConfig config,
                         final ArgumentParser parser,
                         final List<String> adminActionsRequired,
                         final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
          "No configuration change has actually been applied.  The new " +
          "configuration will not take effect until this notification " +
          "manager is disabled and re-enabled or until the server is " +
          "restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this notification manager
   * is to be taken out of service.
   */
  public void finalizeNotificationManager()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
