/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2013-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.types.Entry;
import com.unboundid.directory.sdk.ds.config.
            OneTimePasswordDeliveryMechanismConfig;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.ldap.sdk.unboundidds.extensions.
            SupportedOTPDeliveryMechanismInfo;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which
 * attempt to deliver one-time passwords (OTPs) to end users through some
 * out-of-band mechanism.  The one-time passwords will have been generated by
 * the deliver one-time password extended request, and may be used to
 * authenticate to the server through the UNBOUNDID-DELIVERED-OTP SASL
 * mechanism.
 * <BR>
 * <H2>Configuring One-Time Password Delivery Mechanisms</H2>
 * In order to configure a one-time password delivery mechanism created using
 * this API, use a command like:
 * <PRE>
 *      dsconfig create-otp-delivery-mechanism \
 *           --mechanism-name "<I>{mechanism-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{mechanism-name}</I>" is the name to use for the one-time password
 * instance, "<I>{class-name}</I>" is the fully-qualified name of the Java class
 * that extends
 * {@code com.unboundid.directory.sdk.ds.api.OneTimePasswordDeliveryMechanism},
 * and "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the one-time password delivery mechanism.  If multiple arguments
 * should be provided to the OTP delivery mechanism, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=true)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class OneTimePasswordDeliveryMechanism
       implements UnboundIDExtension,
                  Reconfigurable<OneTimePasswordDeliveryMechanismConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this one-time password delivery mechanism.  All
   * OTP delivery mechanism implementations must include a default constructor,
   * but any initialization should generally be done in the
   * {@code initializeOTPDeliveryMechanism} method.
   */
  public OneTimePasswordDeliveryMechanism()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  @Override()
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  @Override()
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this one-time password delivery mechanism.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this OTP delivery
   *                        mechanism.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this OTP delivery mechanism.
   *
   * @throws  LDAPException  If a problem occurs while initializing this OTP
   *                         delivery mechanism.
   */
  public void initializeOTPDeliveryMechanism(
                   final DirectoryServerContext serverContext,
                   final OneTimePasswordDeliveryMechanismConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public boolean isConfigurationAcceptable(
                      final OneTimePasswordDeliveryMechanismConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public ResultCode applyConfiguration(
                         final OneTimePasswordDeliveryMechanismConfig config,
                         final ArgumentParser parser,
                         final List<String> adminActionsRequired,
                         final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this one-time " +
                "password delivery mechanism is disabled and re-enabled or " +
                "until the server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this one-time password
   * delivery mechanism is to be taken out of service.
   */
  public void finalizeOTPDeliveryMechanism()
  {
    // No implementation is required.
  }



  /**
   * Attempts to deliver the provided one-time password to the specified user.
   *
   * @param  oneTimePassword    The one-time password to be delivered.
   * @param  userEntry          The entry for the user to whom the one-time
   *                            password should be delivered.
   * @param  targetRecipientID  The target recipient ID that should be used for
   *                            the delivery, if possible.  It may be
   *                            {@code null} if there is no appropriate
   *                            identifier for this delivery mechanism or if
   *                            the delivery mechanism may choose which
   *                            recipient ID should be used.  However, if it is
   *                            non-{@code null}, then the delivery mechanism
   *                            must verify that the provided recipient ID is
   *                            valid for the associated user (e.g., for an
   *                            e-mail delivery mechanism, the recipient ID
   *                            would probably be an e-mail address, and the
   *                            delivery mechanism must verify that the provided
   *                            e-mail address is associated with the given
   *                            user account).
   * @param  resultMessage      A buffer to which a message may be appended with
   *                            additional information about the password
   *                            delivery.
   *
   * @return  An identifier for the user in a format appropriate to the delivery
   *          mechanism (e.g., an e-mail address for an e-mail delivery
   *          mechanism, a phone number for an SMS or voice call delivery
   *          mechanism, etc.).  It may be {@code null} if there is no
   *          appropriate identifier for this delivery mechanism that is needed
   *          beyond the name of the delivery mechanism and the DN of the
   *          recipient.
   *
   * @throws  LDAPException  If a problem is encountered while attempting to
   *                         send the one-time password to the user.  The
   *                         exception should have a result code of
   *                         {@code TOKEN_DELIVERY_INVALID_RECIPIENT_ID} if the
   *                         client specified a recipient ID that is not valid
   *                         for the target user,
   *                         {@code TOKEN_DELIVERY_MECHANISM_UNAVAILABLE} if
   *                         this delivery mechanism is not supported for the
   *                         target user, or
   *                         {@code TOKEN_DELIVERY_ATTEMPT_FAILED} if the
   *                         delivery mechanism should be supported but the
   *                         attempt to deliver the one-time password failed.
   */
  public abstract String deliverOneTimePassword(final String oneTimePassword,
                              final Entry userEntry,
                              final String targetRecipientID,
                              final StringBuilder resultMessage)
         throws LDAPException;



  /**
   * Indicates whether this one-time password delivery mechanism supports
   * general purpose message delivery via the {@link #deliverMessage} method.
   *
   * @return  {@code true} if the {@code deliverMessage} method can be used for
   *          general-purpose message delivery, or {@code false} if only the
   *          {@code deliverOneTimePassword} method can be used.
   */
  public boolean supportsGenericMessageDelivery()
  {
    return false;
  }



  /**
   * Attempts to deliver a message (containing a single-use token) to the
   * specified user.
   *
   * @param  config              The general configuration for this OTP delivery
   *                             mechanism.
   * @param  tokenID             The token ID for the single-use token contained
   *                             in the message to be delivered.
   * @param  tokenValue          The value for the single-use token contained
   *                             in the message to be delivered.
   * @param  messageSubject      The message subject that should be used, if
   *                             appropriate.  It may be {@code null} if no
   *                             subject was provided.
   * @param  fullMessage         The message that should be delivered if this
   *                             delivery mechanism does not impose a
   *                             significant constraint on message size.
   * @param  compactMessage      The message that should be delivered if this
   *                             delivery mechanism does impose a significant
   *                             constraint on message size.
   * @param  userEntry           The entry for the user to whom the token should
   *                             be delivered.
   * @param  targetRecipientID   The target recipient ID that should be used for
   *                             the delivery, if possible.  It may be
   *                             {@code null} if there is no appropriate
   *                             identifier for this delivery mechanism or if
   *                             the delivery mechanism may choose which
   *                             recipient ID should be used.  However, if it is
   *                             non-{@code null}, then the delivery mechanism
   *                             must verify that the provided recipient ID is
   *                             valid for the associated user (e.g., for an
   *                             e-mail delivery mechanism, the recipient ID
   *                             would probably be an e-mail address, and the
   *                             delivery mechanism must verify that the
   *                             provided e-mail address is associated with the
   *                             given user account).
   * @param  resultMessage       A buffer to which a message may be appended
   *                             with additional information about the token
   *                             delivery.
   *
   * @return  An identifier for the user in a format appropriate to the delivery
   *          mechanism (e.g., an e-mail address for an e-mail delivery
   *          mechanism, a phone number for an SMS or voice call delivery
   *          mechanism, etc.).  It may be {@code null} if there is no
   *          appropriate identifier for this delivery mechanism that is needed
   *          beyond the name of the delivery mechanism and the DN of the
   *          recipient.
   *
   * @throws  LDAPException  If a problem is encountered while attempting to
   *                         deliver the single-use token message to the user.
   *                         The exception should have a result code of
   *                         {@code TOKEN_DELIVERY_INVALID_RECIPIENT_ID} if the
   *                         client specified a recipient ID that is not valid
   *                         for the target user,
   *                         {@code TOKEN_DELIVERY_MECHANISM_UNAVAILABLE} if
   *                         this delivery mechanism is not supported for the
   *                         target user, or
   *                         {@code TOKEN_DELIVERY_ATTEMPT_FAILED} if the
   *                         delivery mechanism should be supported but the
   *                         attempt to deliver the token failed.
   */
  public String deliverMessage(
                     final OneTimePasswordDeliveryMechanismConfig config,
                     final String tokenID, final String tokenValue,
                     final String messageSubject, final String fullMessage,
                     final String compactMessage, final Entry userEntry,
                     final String targetRecipientID,
                     final StringBuilder resultMessage)
         throws LDAPException
  {
    throw new LDAPException(ResultCode.OTHER,
         "The '" + config.getConfigObjectName() + "' OTP delivery mechanism " +
              "does not support arbitrary message delivery.");
  }



  /**
   * Updates the provided list with one or more
   * {@code SupportedOTPDeliveryMechanismInfo} objects indicating whether this
   * OTP delivery mechanism is supported for the specified user.
   *
   * @param  config                 The general configuration for this OTP
   *                                delivery mechanism.
   * @param  userEntry              The entry of the user for whom to make the
   *                                determination.
   * @param  deliveryMechanismInfo  The list to which any objects may be
   *                                appended with information about the level
   *                                of support this OTP delivery mechanism has
   *                                for use in conjunction with the specified
   *                                user.
   */
  public void getSupportedDeliveryMechanismInfo(
       final OneTimePasswordDeliveryMechanismConfig config,
       final Entry userEntry,
       final List<SupportedOTPDeliveryMechanismInfo> deliveryMechanismInfo)
  {
    deliveryMechanismInfo.add(
         new SupportedOTPDeliveryMechanismInfo(config.getConfigObjectName(),
              null, null));
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
