/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.unboundid.asn1.ASN1OctetString;
import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.ds.config.PasswordStorageSchemeConfig;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.DN;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.ByteString;
import com.unboundid.util.Extensible;
import com.unboundid.util.StaticUtils;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * NOTE:  The {@link EnhancedPasswordStorageScheme} class provides both a
 * simpler and more functional API for interacting with passwords.  It may be
 * desirable for new storage scheme implementations to implement that API rather
 * than this older version.
 * <BR><BR>
 * This class defines an API that must be implemented by extensions which may be
 * used to encode passwords for storage in the server.  Ideally, encoded
 * passwords should be stored in a secure manner so that anyone with access to
 * the encoded password will not be able to determine the clear-text password
 * with which it is associated (e.g., using a one-way message digest, or
 * using reversible encryption with a securely-obtained key).  Passwords are not
 * required to be stored in a reversible form that allows the server to
 * determine the clear-text password used to generate an encoded representation
 * as long as it is possible to determine whether a given clear-text password
 * may be used to generate a provided encoded representation.
 * <BR><BR>
 * Encoded passwords may taken one of two forms.  The first is the "user
 * password" syntax, in which the encoded password is represented by the name of
 * the storage scheme in curly braces followed by the transformed password
 * (e.g., "{scheme}encoded").  This format isn't based on any defined standard,
 * but is commonly used by a number of directory server implementations.  The
 * second format is the authentication password syntax as described in RFC 3112,
 * in which the encoded representation is broken into scheme, authInfo, and
 * authValue segments separated by dollar signs (e.g.,
 * "scheme$authInfo$authValue").  All password storage schemes are required to
 * support the "user password" syntax and may optionally also support the
 * authentication password syntax.
 * <BR>
 * <H2>Configuring Password Storage Schemes</H2>
 * In order to configure a password storage scheme created using this API, use
 * a command like:
 * <PRE>
 *      dsconfig create-password-storage-scheme \
 *           --scheme-name "<I>{scheme-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{scheme-name}</I>" is the name to use for the password storage
 * scheme instance, "<I>{class-name}</I>" is the fully-qualified name of the
 * Java class that extends
 * {@code com.unboundid.directory.sdk.ds.api.PasswordStorageScheme}, and
 * "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the password storage scheme.  If multiple arguments should be
 * provided to the password storage scheme, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 *
 * @see EnhancedPasswordStorageScheme
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=false)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class PasswordStorageScheme
       implements UnboundIDExtension,
                  Reconfigurable<PasswordStorageSchemeConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this password storage scheme.  All password
   * storage scheme implementations must include a default constructor, but any
   * initialization should generally be done in the
   * {@code initializePasswordStorageScheme} method.
   */
  public PasswordStorageScheme()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this password storage scheme.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this password storage
   *                        scheme.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this password storage scheme.
   *
   * @throws  LDAPException  If a problem occurs while initializing this
   *                         password storage scheme.
   */
  public void initializePasswordStorageScheme(
                   final DirectoryServerContext serverContext,
                   final PasswordStorageSchemeConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final PasswordStorageSchemeConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final PasswordStorageSchemeConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this password " +
                "storage scheme is disabled and re-enabled or until the " +
                "server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this password storage
   * scheme is to be taken out of service.
   */
  public void finalizePasswordStorageScheme()
  {
    // No implementation is required.
  }



  /**
   * Retrieves the name for this password storage scheme.  This will be the
   * identifier which appears in curly braces at the beginning of the encoded
   * password.  The name should not include curly braces.
   *
   * @return  The name for this password storage scheme.
   */
  public abstract String getStorageSchemeName();



  /**
   * Indicates whether this password storage scheme encodes passwords in a form
   * that allows the original plaintext value to be obtained from the encoded
   * representation.
   *
   * @return  {@code true} if the original plaintext password may be obtained
   *          from the encoded password, or {@code false} if not.
   */
  public abstract boolean isReversible();



  /**
   * Indicates whether this password storage scheme encodes passwords in a form
   * that may be considered secure.  A storage scheme should only be considered
   * secure if it is not possible to trivially determine a clear-text value
   * which may be used to generate a given encoded representation.
   *
   * @return  {@code true} if this password storage scheme may be considered
   *          secure, or {@code false} if not.
   */
  public abstract boolean isSecure();



  /**
   * Encodes the provided plaintext password.  The encoded password should not
   * include the scheme name in curly braces.
   *
   * @param  plaintext  The plaintext password to be encoded.  It must not be
   *                    {@code null}.  Note that there is no guarantee that
   *                    password validators have yet been invoked for this
   *                    password, so this password storage scheme implementation
   *                    should not make any assumptions about the format of the
   *                    plaintext password or whether it will actually be
   *                    allowed for use in the entry.
   *
   * @return  The encoded representation of the provided password.
   *
   * @throws  LDAPException  If a problem occurs while attempting to encode the
   *                         password.
   */
  public abstract ByteString encodePassword(final ByteString plaintext)
         throws LDAPException;



  /**
   * Encodes the provided plaintext password, prefixing the encoded
   * representation with the name of the storage scheme in curly braces.
   *
   * @param  plaintext  The plaintext password to be encoded.  It must not be
   *                    {@code null}.  Note that there is no guarantee that
   *                    password validators have yet been invoked for this
   *                    password, so this password storage scheme implementation
   *                    should not make any assumptions about the format of the
   *                    plaintext password or whether it will actually be
   *                    allowed for use in the entry.
   *
   * @return  The encoded representation of the provided password, prefixed with
   *          the storage scheme name.
   *
   * @throws  LDAPException  If a problem occurs while attempting to encode the
   *                         password.
   */
  public ByteString encodePasswordWithScheme(final ByteString plaintext)
         throws LDAPException
  {
    final byte[] schemeBytes = StaticUtils.getBytes(getStorageSchemeName());
    final byte[] encodedPW   = encodePassword(plaintext).getValue();

    final byte[] b = new byte[schemeBytes.length + encodedPW.length + 2];

    int pos = 0;
    b[pos++] = '{';

    System.arraycopy(schemeBytes, 0, b, pos, schemeBytes.length);
    pos += schemeBytes.length;

    b[pos++] = '}';
    System.arraycopy(encodedPW, 0, b, pos, encodedPW.length);

    return new ASN1OctetString(b);
  }



  /**
   * Encodes the provided plaintext password, prefixing the encoded
   * representation with the name of the storage scheme in curly braces.  If an
   * entry DN is provided, then a deterministic encoding should be used (e.g.,
   * a salt generated from the provided DN) so that the same encoding will
   * always be used for same plaintext in the same entry.  This will primarily
   * be used when importing an LDIF file containing plaintext passwords so that
   * the same LDIF file can be imported into multiple servers and the same
   * encoded passwords will be generated in each server.
   *
   * @param  plaintext  The plaintext password to be encoded.  It must not be
   *                    {@code null}.  Note that there is no guarantee that
   *                    password validators have yet been invoked for this
   *                    password, so this password storage scheme implementation
   *                    should not make any assumptions about the format of the
   *                    plaintext password or whether it will actually be
   *                    allowed for use in the entry.
   * @param  entryDN    The DN of the entry in which the encoded password will
   *                    appear.  This may be {@code null} if it is not known.
   *
   * @return  The encoded representation of the provided password, prefixed with
   *          the storage scheme name.
   *
   * @throws  LDAPException  If a problem occurs while attempting to encode the
   *                         password.
   */
  public ByteString encodePasswordWithScheme(final ByteString plaintext,
                                             final DN entryDN)
         throws LDAPException
  {
    return encodePasswordWithScheme(plaintext);
  }



  /**
   * Indicates whether the provided plaintext password could have been used to
   * generate the given encoded password.
   *
   * @param  plaintext  The plaintext password for which to make the
   *                    determination.
   * @param  encoded    The encoded password for which to make the
   *                    determination.  It will not include the scheme name.
   *
   * @return  {@code true} if the provided clear-text password could have been
   *          used to generate the encoded password, or {@code false} if not.
   */
  public abstract boolean passwordMatches(final ByteString plaintext,
                                          final ByteString encoded);



  /**
   * Attempts to determine the plaintext password used to generate the provided
   * encoded password.  This method should only be called if the
   * {@link #isReversible} method returns {@code true}.
   *
   * @param  encoded  The encoded password for which to obtain the original
   *                  plaintext password.  It must not be {@code null} and will
   *                  not be prefixed with the scheme name.
   *
   * @return  The plaintext password obtained from the given encoded password.
   *
   * @throws  LDAPException  If this password storage scheme is not reversible,
   *                         or if the provided value could not be decoded to
   *                         its plaintext representation.
   */
  public abstract ByteString getPlaintextValue(final ByteString encoded)
         throws LDAPException;



  /**
   * Indicates whether this password storage scheme provides the ability to
   * encode passwords in the authentication password syntax as described in RFC
   * 3112.
   *
   * @return  {@code true} if this password storage scheme supports the
   *          authentication password syntax, or {@code false} if not.
   */
  public boolean supportsAuthPasswordSyntax()
  {
    return false;
  }



  /**
   * Retrieves the name that should be used to identify this password storage
   * scheme when encoding passwords using the authentication password syntax as
   * described in RFC 3112.  This should only be used if the
   * {@link #supportsAuthPasswordSyntax} method returns {@code true}.
   *
   * @return  The name that should be used to identify this password storage
   *          scheme when encoding passwords using the authentication password
   *          syntax.
   */
  public String getAuthPasswordSchemeName()
  {
    return getStorageSchemeName();
  }



  /**
   * Encodes the provided plaintext password using the authentication password
   * syntax as defined in RFC 3112.  This should only be used if the
   * {@link #supportsAuthPasswordSyntax} method returns {@code true}.
   *
   * @param  plaintext  The plaintext password to be encoded.  It must not be
   *                    {@code null}.  Note that there is no guarantee that
   *                    password validators have yet been invoked for this
   *                    password, so this password storage scheme implementation
   *                    should not make any assumptions about the format of the
   *                    plaintext password or whether it will actually be
   *                    allowed for use in the entry.
   *
   * @return  The encoded representation of the provided password.
   *
   * @throws  LDAPException  If a problem occurs while encoding the provided
   *                         password, or if this password storage scheme does
   *                         not support the authentication password syntax.
   */
  public ByteString encodeAuthPassword(final ByteString plaintext)
         throws LDAPException
  {
    throw new LDAPException(ResultCode.INVALID_ATTRIBUTE_SYNTAX,
         "This password storage scheme does not support the use of the " +
         "authentication password syntax.");
  }



  /**
   * Indicates whether the provided plaintext password may be used to generate
   * an encoded password with the given authInfo and authValue elements when
   * using the authentication password syntax as defined in RFC 3112.  This
   * should only be used if the {@link #supportsAuthPasswordSyntax} method
   * returns {@code true}.
   *
   * @param  plaintext  The plaintext password for which to make the
   *                    determination.
   * @param  authInfo   The authInfo portion of the encoded password for which
   *                    to make the determination.
   * @param  authValue  The authValue portion of the encoded password for which
   *                    to make the determination.
   *
   * @return  {@code true} if the provided plaintext password could be used to
   *          generate an encoded password with the given authInfo and authValue
   *          portions, or {@code false} if not.
   */
  public boolean authPasswordMatches(final ByteString plaintext,
                                     final String authInfo,
                                     final String authValue)
  {
    return false;
  }



  /**
   * Obtains the plaintext password that was used to generate an encoded
   * password with the given authInfo and authValue elements when using the
   * authentication password syntax as described in RFC 3112.  This should only
   * be used if both the {@link #supportsAuthPasswordSyntax} and
   * {@link #isReversible} methods return {@code true}.
   *
   * @param  authInfo   The authInfo portion of the encoded password for which
   *                    to retrieve the corresponding plaintext value.
   * @param  authValue  The authValue portion of the encoded password for which
   *                    to retrieve the corresponding plaintext value.
   *
   * @return  The plaintext password that was used to generate the encoded
   *          password.
   *
   * @throws  LDAPException  If this password storage scheme is not reversible,
   *                         if this password storage scheme does not support
   *                         the authentication password syntax, or if some
   *                         other problem is encountered while attempting to
   *                         determine the plaintext password.
   */
  public ByteString getAuthPasswordPlaintextValue(final String authInfo,
                                                  final String authValue)
         throws LDAPException
  {
    throw new LDAPException(ResultCode.INVALID_ATTRIBUTE_SYNTAX,
         "This password storage scheme does not support the use of the " +
         "authentication password syntax.");
  }



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
