/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.types.Entry;
import com.unboundid.directory.sdk.common.types.OperationContext;
import com.unboundid.directory.sdk.ds.config.PasswordValidatorConfig;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.ldap.sdk.unboundidds.extensions.PasswordQualityRequirement;
import com.unboundid.util.ByteString;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which
 * attempt to determine whether a proposed user password is acceptable.  Each
 * server password policy may be configured with zero or more password
 * validators, and whenever a user changes his or her password (and optionally
 * whenever an administrator resets the password for another user), then each of
 * the password validators configured in the password policy for the target user
 * will be given access to the clear-text password in order to determine whether
 * that password will be allowed.  Password validators will also have access to
 * the rest of the user entry, and may also have access to a clear-text version
 * of the user's current password(s) if they were provided in the request.
 * <BR>
 * <H2>Configuring Password Validators</H2>
 * In order to configure a password validator created using this API, use a
 * command like:
 * <PRE>
 *      dsconfig create-password-validator \
 *           --validator-name "<I>{validator-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{validator-name}</I>" is the name to use for the password validator
 * instance, "<I>{class-name}</I>" is the fully-qualified name of the Java class
 * that extends {@code com.unboundid.directory.sdk.ds.api.PasswordValidator},
 * and "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the password validator.  If multiple arguments should be provided
 * to the password validator, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 *
 * @see  com.unboundid.directory.sdk.ds.scripting.ScriptedPasswordValidator
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=false)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class PasswordValidator
       implements UnboundIDExtension, Reconfigurable<PasswordValidatorConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this password validator.  All password validator
   * implementations must include a default constructor, but any initialization
   * should generally be done in the {@code initializePasswordValidator} method.
   */
  public PasswordValidator()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this password validator.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this password
   *                        validator.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this password validator.
   *
   * @throws  LDAPException  If a problem occurs while initializing this
   *                         password validator.
   */
  public void initializePasswordValidator(
                   final DirectoryServerContext serverContext,
                   final PasswordValidatorConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(final PasswordValidatorConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final PasswordValidatorConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this password " +
                "validator is disabled and re-enabled or until the server " +
                "is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this password validator is
   * to be taken out of service.
   */
  public void finalizePasswordValidator()
  {
    // No implementation is required.
  }



  /**
   * Indicates whether this password validator should be invoked for add
   * operations that attempt to create an entry containing one or more
   * password values.
   *
   * @return  {@code true} if this password validator should be invoked for
   *          add operations that include one or more passwords, or
   *          {@code false} if not.
   */
  public boolean invokeForAdd()
  {
    return true;
  }



  /**
   * Indicates whether this password validator should be invoked for modify or
   * password modify operations that represent a user's attempt to change
   * his/her own password.
   *
   * @return  {@code true} if this password validator should be invoked for
   *          self password change operations, or {@code false} if not.
   */
  public boolean invokeForSelfChange()
  {
    return true;
  }



  /**
   * Indicates whether this password validator should be invoked for modify or
   * password modify operations that represent one user's attempt to change the
   * password for another user.
   *
   * @return  {@code true} if this password validator should be invoked for
   *          administrative password reset operations, or {@code false} if not.
   */
  public boolean invokeForAdministrativeReset()
  {
    return true;
  }



  /**
   * Retrieves the password quality requirement for this password validator, if
   * available.
   *
   * @return  The password quality requirement for this password validator, or
   *          {@code null} if no requirement information is available.
   */
  public PasswordQualityRequirement getPasswordQualityRequirement()
  {
    return null;
  }



  /**
   * Indicates whether the proposed password is acceptable for the specified
   * user.
   *
   * @param  operationContext  The operation context for the associated request.
   *                           It may be associated with an add, modify, or
   *                           password modify operation.
   * @param  newPassword       The proposed new password for the user that
   *                           should be validated.  It will not be encoded or
   *                           obscured in any way.
   * @param  currentPasswords  The current set of passwords for the user, if
   *                           available.  It may be {@code null} if this is
   *                           not available.  Note that even if one or more
   *                           current passwords are available, it may not
   *                           constitute the complete set of passwords for the
   *                           user.
   * @param  userEntry         The entry for the user whose password is being
   *                           changed.
   * @param  invalidReason     A buffer to which a message may be appended to
   *                           indicate why the proposed password is not
   *                           acceptable.
   *
   * @return  {@code true} if the proposed new password is acceptable, or
   *          {@code false} if not.
   */
  public abstract boolean isPasswordAcceptable(
                               final OperationContext operationContext,
                               final ByteString newPassword,
                               final Set<ByteString> currentPasswords,
                               final Entry userEntry,
                               final StringBuilder invalidReason);



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
