/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.unboundid.directory.sdk.broker.internal.BrokerExtension;
import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.operation.AddRequest;
import com.unboundid.directory.sdk.common.operation.AddResult;
import com.unboundid.directory.sdk.common.operation.BindResult;
import com.unboundid.directory.sdk.common.operation.CompareRequest;
import com.unboundid.directory.sdk.common.operation.CompareResult;
import com.unboundid.directory.sdk.common.operation.DeleteRequest;
import com.unboundid.directory.sdk.common.operation.DeleteResult;
import com.unboundid.directory.sdk.common.operation.ExtendedRequest;
import com.unboundid.directory.sdk.common.operation.ExtendedResult;
import com.unboundid.directory.sdk.common.operation.ModifyRequest;
import com.unboundid.directory.sdk.common.operation.ModifyResult;
import com.unboundid.directory.sdk.common.operation.ModifyDNRequest;
import com.unboundid.directory.sdk.common.operation.ModifyDNResult;
import com.unboundid.directory.sdk.common.operation.SASLBindRequest;
import com.unboundid.directory.sdk.common.operation.SearchRequest;
import com.unboundid.directory.sdk.common.operation.SearchResult;
import com.unboundid.directory.sdk.common.operation.SimpleBindRequest;
import com.unboundid.directory.sdk.common.operation.UpdatableAbandonRequest;
import com.unboundid.directory.sdk.common.operation.UpdatableAddRequest;
import com.unboundid.directory.sdk.common.operation.UpdatableAddResult;
import com.unboundid.directory.sdk.common.operation.UpdatableBindResult;
import com.unboundid.directory.sdk.common.operation.UpdatableCompareRequest;
import com.unboundid.directory.sdk.common.operation.UpdatableCompareResult;
import com.unboundid.directory.sdk.common.operation.UpdatableDeleteRequest;
import com.unboundid.directory.sdk.common.operation.UpdatableDeleteResult;
import com.unboundid.directory.sdk.common.operation.UpdatableExtendedRequest;
import com.unboundid.directory.sdk.common.operation.UpdatableExtendedResult;
import com.unboundid.directory.sdk.common.operation.UpdatableGenericResult;
import com.unboundid.directory.sdk.common.operation.UpdatableModifyRequest;
import com.unboundid.directory.sdk.common.operation.UpdatableModifyResult;
import com.unboundid.directory.sdk.common.operation.UpdatableModifyDNRequest;
import com.unboundid.directory.sdk.common.operation.UpdatableModifyDNResult;
import com.unboundid.directory.sdk.common.operation.UpdatableSASLBindRequest;
import com.unboundid.directory.sdk.common.operation.UpdatableSearchRequest;
import com.unboundid.directory.sdk.common.operation.UpdatableSearchResult;
import com.unboundid.directory.sdk.common.operation.UpdatableSimpleBindRequest;
import com.unboundid.directory.sdk.common.operation.UpdatableUnbindRequest;
import com.unboundid.directory.sdk.common.types.ActiveOperationContext;
import com.unboundid.directory.sdk.common.types.ActiveSearchOperationContext;
import com.unboundid.directory.sdk.common.types.CompletedOperationContext;
import com.unboundid.directory.sdk.common.types.CompletedSearchOperationContext;
import com.unboundid.directory.sdk.common.types.ClientContext;
import com.unboundid.directory.sdk.common.types.DisconnectReason;
import com.unboundid.directory.sdk.common.types.Entry;
import com.unboundid.directory.sdk.common.types.UpdatableEntry;
import com.unboundid.directory.sdk.ds.config.PluginConfig;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.ds.types.IntermediateResponsePluginResult;
import com.unboundid.directory.sdk.ds.types.LDIFPluginResult;
import com.unboundid.directory.sdk.ds.types.PostConnectPluginResult;
import com.unboundid.directory.sdk.ds.types.PostDisconnectPluginResult;
import com.unboundid.directory.sdk.ds.types.PostOperationPluginResult;
import com.unboundid.directory.sdk.ds.types.PostResponsePluginResult;
import com.unboundid.directory.sdk.ds.types.PreOperationPluginResult;
import com.unboundid.directory.sdk.ds.types.PreParsePluginResult;
import com.unboundid.directory.sdk.ds.types.SearchEntryPluginResult;
import com.unboundid.directory.sdk.ds.types.SearchReferencePluginResult;
import com.unboundid.directory.sdk.ds.types.StartupDependency;
import com.unboundid.directory.sdk.ds.types.StartupPluginResult;
import com.unboundid.directory.sdk.ds.types.SubordinateModifyDNPluginResult;
import com.unboundid.directory.sdk.metrics.internal.MetricsEngineExtension;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.Control;
import com.unboundid.ldap.sdk.IntermediateResponse;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.Modification;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which act as
 * server plugins.  There are several different kinds of plugins which may be
 * invoked for different purposes, including:
 * <UL>
 *   <LI>startup -- Startup plugins may be used to perform custom processing
 *       when the server is in the process of staring up.  By default, the
 *       processing will be performed near the end of the startup phase just
 *       before the server begins to accept connections from external clients,
 *       but if one or more startup dependencies are defined, then the plugin
 *       may be invoked earlier in the startup process as soon as all declared
 *       dependencies have been satisfied.</LI>
 *   <LI>shutdown -- Shutdown plugins may be used to perform custom processing
 *       when the server is in the process of shutting down.  This will occur
 *       near the beginning of the shutdown process, just after the server stops
 *       accepting new client connections and terminates existing connections
 *       but before shutting down any other components.  Note that shutdown
 *       plugins may not always be invoked in some unusual shutdown scenarios
 *       (e.g., if the server process is forcefully killed, or in the event of a
 *       hardware, OS, or JVM failure).</LI>
 *   <LI>LDIF import -- LDIF import plugins are invoked for each entry read from
 *       an LDIF file to be imported into a backend.  It is possible to alter
 *       the contents of the entry or to cause the entry to be excluded from the
 *       import.</LI>
 *   <LI>LDIF export -- LDIF export plugins are invoked for each entry exported
 *       from a backend to be written to an LDIF file.  It is possible to alter
 *       the contents of the entry or to cause the entry to be excluded from the
 *       export.</LI>
 *   <LI>post-connect -- Post-connect plugins are invoked during the course of
 *       accepting a new client connection.  They may access information about
 *       the client and may optionally terminate the client connection if it is
 *       deemed appropriate.</LI>
 *   <LI>post-disconnect -- Post-disconnect plugins are invoked after a client
 *       connection has been closed, regardless of whether the closure was
 *       initiated by the client (e.g., because of an unbind request or simply
 *       closing the socket) or by the server (e.g., because the connection had
 *       been idle for too long or because the client violated some server-side
 *       constraint.</LI>
 *   <LI>pre-parse -- Pre-parse plugins are invoked just after the server has
 *       received an abandon, add, bind, compare, delete, extended, modify,
 *       modify DN, search, or unbind request.  They may be used to obtain
 *       information about the requests, alter the contents of the request or
 *       prevent the server from processing the request.
 *   <LI>pre-operation -- Pre-operation plugins are invoked for add, bind,
 *       compare, delete, extended, modify, modify DN, and search operations
 *       after some level of validation has been performed but just before the
 *       core processing for the operation.  They may not be used to alter the
 *       request, but they may still be used to obtain information about the
 *       request or to prevent the operation from being processed.</LI>
 *   <LI>post-operation -- Post-operation plugins are invoked for add, bind,
 *       compare, delete, extended, modify, modify DN, and search operations
 *       after most processing has completed for the operation but before the
 *       result has been returned to the client.  They may be used to obtain
 *       information about the result or to alter the contents of the result to
 *       be returned.</LI>
 *   <LI>post-response -- Post-response plugins are invoked for add, bind,
 *       compare, delete, extended, modify, modify DN, and search operations
 *       after the response has already been sent to the client.  They may be
 *       used to perform processing for operations without delaying the response
 *       to the client.</LI>
 *   <LI>post-replication -- Post-replication plugins are invoked for add,
 *       delete, modify, and modify DN operations that have been received and
 *       processed via replication.  They may be used to obtain information or
 *       perform processing for replicated changes, but may not alter those
 *       changes.</LI>
 *   <LI>search result entry -- Search result entry plugins are invoked for each
 *       entry to be returned to the client during the course of processing a
 *       search operation.  They may obtain information about the entry to be
 *       returned, alter the entry contents, or prevent the entry from being
 *       returned.</LI>
 *   <LI>search result reference -- Search result reference plugins are invoked
 *       for each reference to be returned to the client during the course of
 *       processing a search operation.  They may obtain information about the
 *       reference to be returned, alter the referral URLs to be returned, or
 *       prevent the reference from being returned.</LI>
 *   <LI>subordinate modify DN -- Subordinate modify DN plugins are invoked for
 *       each entry below the target entry being renamed in the course of a
 *       modify DN operation.  They may be used to obtain information about the
 *       subordinate entry to be renamed and optionally to alter the contents
 *       of that entry.</LI>
 *   <LI>intermediate response -- Intermediate response plugins are invoked for
 *       each intermediate response to be returned to the client.  They may be
 *       used to obtain information about the response or to prevent it from
 *       being returned to the client.</LI>
 * </UL>
 * <BR><BR>
 * A single plugin instance may be configured to act in any combination of these
 * contexts.  For example, a plugin may be configured to operate during both
 * LDIF import and pre-parse add contexts so that it can transform the contents
 * of entries whether regardless of the way they are inserted into the backend.
 * <BR>
 * <H2>Configuring Plugins</H2>
 * In order to configure a plugin created using this API, use a command like:
 * <PRE>
 *      dsconfig create-plugin \
 *           --plugin-name "<I>{plugin-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set plugin-type:{plugin-type} \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{plugin-name}</I>" is the name to use for the plugin instance,
 * "<I>{plugin-type}</I>" is the name of a plugin type for which the plugin
 * should be invoked,  "<I>{class-name}</I>" is the fully-qualified name of the
 * Java class that extends {@code com.unboundid.directory.sdk.ds.api.Plugin},
 * and "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the plugin.  If the plugin should be invoked for multiple plugin
 * types, then the "<CODE>--set plugin-type:<I>{plugin-type}</I></CODE>"
 * option should be provided multiple times.  Similarly, if multiple arguments
 * should be provided to the plugin, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 *
 * @see  com.unboundid.directory.sdk.ds.scripting.ScriptedPlugin
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=true,
     notes="Some plugin types will not be invoked for proxied operations, " +
          "including pre-operation, post-operation, search result entry, " +
          "search result reference, intermediate response, and subordinate " +
          "modify DN.  A proxy transformation may be used to achieve the " +
          "same result in many cases.")
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false,
     notes="This extension type is primarily useful within the " +
          "${SYNC_SERVER_BASE_NAME} to have custom code run at server " +
          "startup or shutdown using the 'startup' and 'shutdown' plugin " +
          "types.")
@MetricsEngineExtension(
     notes="This extension type is primarily useful within the " +
          "${METRICS_ENGINE_BASE_NAME} to have custom code run at server " +
          "startup or shutdown using the 'startup' and 'shutdown' plugin " +
          "types.")
@BrokerExtension(
     notes="This extension type is primarily useful within the " +
          "${BROKER_BASE_NAME} to have custom code run at server startup " +
          "or shutdown using the 'startup' and 'shutdown' plugin types.")
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class Plugin
       implements UnboundIDExtension, Reconfigurable<PluginConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this plugin.  All plugin implementations must
   * include a default constructor, but any initialization should generally be
   * done in the {@code initializePlugin} method.
   */
  public Plugin()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this plugin.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this plugin.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this plugin.
   *
   * @throws  LDAPException  If a problem occurs while initializing this plugin.
   */
  public void initializePlugin(final DirectoryServerContext serverContext,
                               final PluginConfig config,
                               final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(final PluginConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final PluginConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this plugin is " +
                "disabled and re-enabled or until the server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this plugin is to be taken
   * out of service.
   */
  public void finalizePlugin()
  {
    // No implementation is required.
  }



  /**
   * Retrieves the set of startup dependencies which must be resolved in order
   * for this plugin to be invoked at server startup.  This is only applicable
   * for startup plugins, in which case it may be possible to have the plugin
   * startup processing invoked as early as possible.  If it returns
   * {@code null} then startup processing for this plugin will be performed at
   * the default time during startup.
   *
   * @return  The set of startup dependencies for this plugin, or {@code null}
   *          if it is not a startup plugin, or if startup processing should be
   *          invoked at the default time during server startup.
   */
  public Set<StartupDependency> getStartupDependencies()
  {
    return null;
  }



  /**
   * Performs any processing which may be necessary when the server is starting.
   *
   * @return  Information about the result of the plugin processing.
   */
  public StartupPluginResult doStartup()
  {
    // No processing is performed by default.
    return StartupPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary when the server is shutting
   * down.
   *
   * @param  shutdownReason  A message which may provide information about the
   *                         reason the server is shutting down.
   */
  public void doShutdown(final String shutdownReason)
  {
    // No processing is performed by default.
  }



  /**
   * Performs any processing which may be necessary for the provided entry to
   * be imported into the server.
   *
   * @param  entry  The entry to be imported.  It may be altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public LDIFPluginResult doLDIFImport(final UpdatableEntry entry)
  {
    // No processing is performed by default.
    return LDIFPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary for the provided entry to
   * be exported from the server.
   *
   * @param  entry  The entry to be exported.  It may be altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public LDIFPluginResult doLDIFExport(final UpdatableEntry entry)
  {
    // No processing is performed by default.
    return LDIFPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary when the server has accepted
   * a new client connection.
   *
   * @param  clientContext  Information about the client connection that has
   *                        been established.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostConnectPluginResult doPostConnect(
                                      final ClientContext clientContext)
  {
    // No processing is performed by default.
    return PostConnectPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary when the server has
   * terminated a client connection.
   *
   * @param  clientContext     Information about the client connection that has
   *                           been established.
   * @param  disconnectReason  A general reason for the disconnect.
   * @param  message           A message which may provide additional
   *                           information about the disconnect.  It may be
   *                           {@code null} if none is available.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostDisconnectPluginResult doPostDisconnect(
              final ClientContext clientContext,
              final DisconnectReason disconnectReason, final String message)
  {
    // No processing is performed by default.
    return PostDisconnectPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server starts
   * processing for an abandon request.
   *
   * @param  operationContext  The context for the abandon operation.
   * @param  request           The abandon request to be processed.  It may be
   *                           altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreParsePluginResult doPreParse(
              final ActiveOperationContext operationContext,
              final UpdatableAbandonRequest request)
  {
    // No processing is performed by default.
    return PreParsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server starts
   * processing for an add request.  This will be invoked only for add
   * operations requested directly by clients, but not for add operations
   * received from another server via replication.
   *
   * @param  operationContext  The context for the add operation.
   * @param  request           The add request to be processed.  It may be
   *                           altered if desired.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreParsePluginResult doPreParse(
              final ActiveOperationContext operationContext,
              final UpdatableAddRequest request,
              final UpdatableAddResult result)
  {
    // No processing is performed by default.
    return PreParsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server actually
   * attempts to add an entry to the appropriate backend.  This will be invoked
   * only for add operations requested directly by clients, but not for add
   * operations received from another server via replication.
   *
   * @param  operationContext  The context for the add operation.
   * @param  request           The add request to be processed.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreOperationPluginResult doPreOperation(
              final ActiveOperationContext operationContext,
              final AddRequest request, final UpdatableAddResult result)
  {
    // No processing is performed by default.
    return PreOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server sends a
   * response for an add operation.  This will be invoked only for add
   * operations requested directly by clients, but not for add operations
   * received from another server via replication.
   *
   * @param  operationContext  The context for the add operation.
   * @param  request           The add request that was processed.
   * @param  result            The result to be returned to the client.  It may
   *                           be altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostOperationPluginResult doPostOperation(
              final ActiveOperationContext operationContext,
              final AddRequest request, final UpdatableAddResult result)
  {
    // No processing is performed by default.
    return PostOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary after all other processing
   * has been completed for an add operation and the response has been sent to
   * the client.  This will be invoked only for add operations requested
   * directly by clients, but not for add operations received from another
   * server via replication.
   *
   * @param  operationContext  The context for the add operation.
   * @param  request           The add request that was processed.
   * @param  result            The result that was returned to the client.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostResponsePluginResult doPostResponse(
              final CompletedOperationContext operationContext,
              final AddRequest request, final AddResult result)
  {
    // No processing is performed by default.
    return PostResponsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary after all other processing
   * has been completed for an add operation which has been received from
   * another server via replication.
   *
   * @param  operationContext  The context for the add operation.
   * @param  request           The add request that was processed.
   * @param  result            The result that was returned to the client.
   */
  public void doPostReplication(
                   final CompletedOperationContext operationContext,
                   final AddRequest request, final AddResult result)
  {
    // No processing is performed by default.
  }



  /**
   * Performs any processing which may be necessary before the server starts
   * processing for a simple bind request.
   *
   * @param  operationContext  The context for the bind operation.
   * @param  request           The bind request to be processed.  It may be
   *                           altered if desired.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreParsePluginResult doPreParse(
              final ActiveOperationContext operationContext,
              final UpdatableSimpleBindRequest request,
              final UpdatableBindResult result)
  {
    // No processing is performed by default.
    return PreParsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server actually
   * attempts to perform the authentication for a simple bind request.
   *
   * @param  operationContext  The context for the bind operation.
   * @param  request           The bind request to be processed.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreOperationPluginResult doPreOperation(
              final ActiveOperationContext operationContext,
              final SimpleBindRequest request,
              final UpdatableBindResult result)
  {
    // No processing is performed by default.
    return PreOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server sends a
   * response for a simple bind operation.
   *
   * @param  operationContext  The context for the bind operation.
   * @param  request           The bind request that was processed.
   * @param  result            The result to be returned to the client.  It may
   *                           be altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostOperationPluginResult doPostOperation(
              final ActiveOperationContext operationContext,
              final SimpleBindRequest request, final UpdatableBindResult result)
  {
    // No processing is performed by default.
    return PostOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary after all other processing
   * has been completed for a simple bind operation and the response has been
   * sent to the client.
   *
   * @param  operationContext  The context for the bind operation.
   * @param  request           The bind request that was processed.
   * @param  result            The result that was returned to the client.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostResponsePluginResult doPostResponse(
              final CompletedOperationContext operationContext,
              final SimpleBindRequest request, final BindResult result)
  {
    // No processing is performed by default.
    return PostResponsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server starts
   * processing for a SASL bind request.
   *
   * @param  operationContext  The context for the bind operation.
   * @param  request           The bind request to be processed.  It may be
   *                           altered if desired.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreParsePluginResult doPreParse(
              final ActiveOperationContext operationContext,
              final UpdatableSASLBindRequest request,
              final UpdatableBindResult result)
  {
    // No processing is performed by default.
    return PreParsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server actually
   * attempts to perform the authentication for a SASL bind request.
   *
   * @param  operationContext  The context for the bind operation.
   * @param  request           The bind request to be processed.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreOperationPluginResult doPreOperation(
              final ActiveOperationContext operationContext,
              final SASLBindRequest request, final UpdatableBindResult result)
  {
    // No processing is performed by default.
    return PreOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server sends a
   * response for a SASL bind operation.
   *
   * @param  operationContext  The context for the bind operation.
   * @param  request           The bind request that was processed.
   * @param  result            The result to be returned to the client.  It may
   *                           be altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostOperationPluginResult doPostOperation(
              final ActiveOperationContext operationContext,
              final SASLBindRequest request, final UpdatableBindResult result)
  {
    // No processing is performed by default.
    return PostOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary after all other processing
   * has been completed for a SASL bind operation and the response has been
   * sent to the client.
   *
   * @param  operationContext  The context for the bind operation.
   * @param  request           The bind request that was processed.
   * @param  result            The result that was returned to the client.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostResponsePluginResult doPostResponse(
              final CompletedOperationContext operationContext,
              final SASLBindRequest request, final BindResult result)
  {
    // No processing is performed by default.
    return PostResponsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server starts
   * processing for a compare request.
   *
   * @param  operationContext  The context for the compare operation.
   * @param  request           The compare request to be processed.  It may be
   *                           altered if desired.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreParsePluginResult doPreParse(
              final ActiveOperationContext operationContext,
              final UpdatableCompareRequest request,
              final UpdatableCompareResult result)
  {
    // No processing is performed by default.
    return PreParsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server actually
   * attempts to perform the core processing for the compare.
   *
   * @param  operationContext  The context for the compare operation.
   * @param  request           The compare request to be processed.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   * @param  entry             The entry targeted by the compare operation.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreOperationPluginResult doPreOperation(
              final ActiveOperationContext operationContext,
              final CompareRequest request, final UpdatableCompareResult result,
              final Entry entry)
  {
    // No processing is performed by default.
    return PreOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server sends a
   * response for a compare operation.
   *
   * @param  operationContext  The context for the compare operation.
   * @param  request           The compare request that was processed.
   * @param  result            The result to be returned to the client.  It may
   *                           be altered if desired.
   * @param  entry             The entry targeted by the compare operation, if
   *                           it exists.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostOperationPluginResult doPostOperation(
              final ActiveOperationContext operationContext,
              final CompareRequest request, final UpdatableCompareResult result,
              final Entry entry)
  {
    // No processing is performed by default.
    return PostOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary after all other processing
   * has been completed for a compare operation and the response has been sent
   * to the client.
   *
   * @param  operationContext  The context for the compare operation.
   * @param  request           The compare request that was processed.
   * @param  result            The result that was returned to the client.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostResponsePluginResult doPostResponse(
              final CompletedOperationContext operationContext,
              final CompareRequest request, final CompareResult result)
  {
    // No processing is performed by default.
    return PostResponsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server starts
   * processing for a delete request.  This will be invoked only for delete
   * operations requested directly by clients, but not for delete operations
   * received from another server via replication.
   *
   * @param  operationContext  The context for the delete operation.
   * @param  request           The delete request to be processed.  It may be
   *                           altered if desired.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreParsePluginResult doPreParse(
              final ActiveOperationContext operationContext,
              final UpdatableDeleteRequest request,
              final UpdatableDeleteResult result)
  {
    // No processing is performed by default.
    return PreParsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server actually
   * attempts to remove the entry from the server.  This will be invoked only
   * for delete operations requested directly by clients, but not for delete
   * operations received from another server via replication.

   *
   * @param  operationContext  The context for the delete operation.
   * @param  request           The delete request to be processed.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   * @param  entry             The entry targeted by the delete operation.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreOperationPluginResult doPreOperation(
              final ActiveOperationContext operationContext,
              final DeleteRequest request, final UpdatableDeleteResult result,
              final Entry entry)
  {
    // No processing is performed by default.
    return PreOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server sends a
   * response for a delete operation.  This will be invoked only for delete
   * operations requested directly by clients, but not for delete operations
   * received from another server via replication.

   *
   * @param  operationContext  The context for the delete operation.
   * @param  request           The delete request that was processed.
   * @param  result            The result to be returned to the client.  It may
   *                           be altered if desired.
   * @param  entry             The entry targeted by the delete operation, if
   *                           it exists.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostOperationPluginResult doPostOperation(
              final ActiveOperationContext operationContext,
              final DeleteRequest request, final UpdatableDeleteResult result,
              final Entry entry)
  {
    // No processing is performed by default.
    return PostOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary after all other processing
   * has been completed for a delete operation and the response has been sent
   * to the client.  This will be invoked only for delete operations requested
   * directly by clients, but not for delete operations received from another
   * server via replication.

   *
   * @param  operationContext  The context for the delete operation.
   * @param  request           The delete request that was processed.
   * @param  result            The result that was returned to the client.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostResponsePluginResult doPostResponse(
              final CompletedOperationContext operationContext,
              final DeleteRequest request, final DeleteResult result)
  {
    // No processing is performed by default.
    return PostResponsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary after all other processing
   * has been completed for a delete operation which has been received from
   * another server via replication.
   *
   * @param  operationContext  The context for the delete operation.
   * @param  request           The delete request that was processed.
   * @param  result            The result that was returned to the client.
   */
  public void doPostReplication(
                   final CompletedOperationContext operationContext,
                   final DeleteRequest request, final DeleteResult result)
  {
    // No processing is performed by default.
  }



  /**
   * Performs any processing which may be necessary before the server starts
   * processing for an extended request.
   *
   * @param  operationContext  The context for the extended operation.
   * @param  request           The extended request to be processed.  It may be
   *                           altered if desired.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreParsePluginResult doPreParse(
              final ActiveOperationContext operationContext,
              final UpdatableExtendedRequest request,
              final UpdatableExtendedResult result)
  {
    // No processing is performed by default.
    return PreParsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server actually
   * attempts to perform the core processing for the extended operation.
   *
   * @param  operationContext  The context for the extended operation.
   * @param  request           The extended request to be processed.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreOperationPluginResult doPreOperation(
              final ActiveOperationContext operationContext,
              final ExtendedRequest request,
              final UpdatableExtendedResult result)
  {
    // No processing is performed by default.
    return PreOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server sends a
   * response for an extended operation.
   *
   * @param  operationContext  The context for the extended operation.
   * @param  request           The extended request that was processed.
   * @param  result            The result to be returned to the client.  It may
   *                           be altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostOperationPluginResult doPostOperation(
              final ActiveOperationContext operationContext,
              final ExtendedRequest request,
              final UpdatableExtendedResult result)
  {
    // No processing is performed by default.
    return PostOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary after all other processing
   * has been completed for an extended operation and the response has been sent
   * to the client.
   *
   * @param  operationContext  The context for the extended operation.
   * @param  request           The extended request that was processed.
   * @param  result            The result that was returned to the client.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostResponsePluginResult doPostResponse(
              final CompletedOperationContext operationContext,
              final ExtendedRequest request, final ExtendedResult result)
  {
    // No processing is performed by default.
    return PostResponsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server starts
   * processing for a modify request.  This will be invoked only for modify
   * operations requested directly by clients, but not for modify operations
   * received from another server via replication.
   *
   * @param  operationContext  The context for the modify operation.
   * @param  request           The modify request to be processed.  It may be
   *                           altered if desired.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreParsePluginResult doPreParse(
              final ActiveOperationContext operationContext,
              final UpdatableModifyRequest request,
              final UpdatableModifyResult result)
  {
    // No processing is performed by default.
    return PreParsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server actually
   * attempts to update the entry in the backend.  This will be invoked only for
   * modify operations requested directly by clients, but not for modify
   * operations received from another server via replication.
   *
   * @param  operationContext  The context for the modify operation.
   * @param  request           The modify request to be processed.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   * @param  oldEntry          The entry as it appeared before the modifications
   *                           were applied.
   * @param  newEntry          The updated entry as it will appear after the
   *                           modifications have been applied.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreOperationPluginResult doPreOperation(
              final ActiveOperationContext operationContext,
              final ModifyRequest request, final UpdatableModifyResult result,
              final Entry oldEntry, final Entry newEntry)
  {
    // No processing is performed by default.
    return PreOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server sends a
   * response for a modify operation.  This will be invoked only for modify
   * operations requested directly by clients, but not for modify operations
   * received from another server via replication.
   *
   * @param  operationContext  The context for the modify operation.
   * @param  request           The modify request that was processed.
   * @param  result            The result to be returned to the client.  It may
   *                           be altered if desired.
   * @param  oldEntry          The entry as it appeared before the modifications
   *                           were applied, if it exists. .
   * @param  newEntry          The entry as it appears after the modifications
   *                           have been applied, if it exists. .
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostOperationPluginResult doPostOperation(
              final ActiveOperationContext operationContext,
              final ModifyRequest request, final UpdatableModifyResult result,
              final Entry oldEntry, final Entry newEntry)
  {
    // No processing is performed by default.
    return PostOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary after all other processing
   * has been completed for a modify operation and the response has been sent
   * to the client.  This will be invoked only for modify operations requested
   * directly by clients, but not for modify operations received from another
   * server via replication.
   *
   * @param  operationContext  The context for the modify operation.
   * @param  request           The modify request that was processed.
   * @param  result            The result that was returned to the client.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostResponsePluginResult doPostResponse(
              final CompletedOperationContext operationContext,
              final ModifyRequest request, final ModifyResult result)
  {
    // No processing is performed by default.
    return PostResponsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary after all other processing
   * has been completed for a modify operation which has been received from
   * another server via replication.
   *
   * @param  operationContext  The context for the modify operation.
   * @param  request           The modify request that was processed.
   * @param  result            The result that was returned to the client.
   */
  public void doPostReplication(
                   final CompletedOperationContext operationContext,
                   final ModifyRequest request, final ModifyResult result)
  {
    // No processing is performed by default.
  }



  /**
   * Performs any processing which may be necessary before the server starts
   * processing for a modify DN request.  This will be invoked only for modify
   * DN operations requested directly by clients, but not for modify DN
   * operations received from another server via replication.
   *
   * @param  operationContext  The context for the modify DN operation.
   * @param  request           The modify DN request to be processed.  It may be
   *                           altered if desired.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreParsePluginResult doPreParse(
              final ActiveOperationContext operationContext,
              final UpdatableModifyDNRequest request,
              final UpdatableModifyDNResult result)
  {
    // No processing is performed by default.
    return PreParsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server actually
   * attempts to update the entry in the backend.  This will be invoked only for
   * modify DN operations requested directly by clients, but not for modify DN
   * operations received from another server via replication.
   *
   * @param  operationContext  The context for the modify DN operation.
   * @param  request           The modify DN request to be processed.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   * @param  oldEntry          The entry as it appeared before being renamed.
   * @param  newEntry          The updated entry as it will appear after it has
   *                           been renamed.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreOperationPluginResult doPreOperation(
              final ActiveOperationContext operationContext,
              final ModifyDNRequest request,
              final UpdatableModifyDNResult result, final Entry oldEntry,
              final Entry newEntry)
  {
    // No processing is performed by default.
    return PreOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary during the course of
   * renaming an entry which is subordinate to an entry targeted by a modify DN
   * operation.  This will be invoked only for both modify DN operations
   * requested by clients and for modify DN operations received from another
   * server via replication.
   *
   * @param  operationContext         The context for the modify DN operation.
   * @param  request                  The modify DN request being processed.
   * @param  result                   The result that will be returned to the
   *                                  client if the plugin result indicates that
   *                                  processing on the operation should be
   *                                  interrupted.  It may be altered if
   *                                  desired.
   * @param  oldSubordinateEntry      The subordinate entry as it appeared
   *                                  before being renamed.
   * @param  newSubordinateEntry      The subordinate entry as it will appear
   *                                  after being renamed.
   * @param  additionalModifications  A list of additional modifications that
   *                                  should be applied to the entry as it is
   *                                  renamed.  This list may be altered if
   *                                  desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public SubordinateModifyDNPluginResult doSubordinateModifyDN(
              final ActiveOperationContext operationContext,
              final ModifyDNRequest request,
              final UpdatableModifyDNResult result,
              final Entry oldSubordinateEntry,
              final Entry newSubordinateEntry,
              final List<Modification> additionalModifications)
  {
    // No processing is performed by default.
    return SubordinateModifyDNPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server sends a
   * response for a modify DN operation.  This will be invoked only for modify
   * DN operations requested directly by clients, but not for modify DN
   * operations received from another server via replication.
   *
   * @param  operationContext  The context for the modify DN operation.
   * @param  request           The modify DN request that was processed.
   * @param  result            The result to be returned to the client.  It may
   *                           be altered if desired.
   * @param  oldEntry          The entry as it appeared before it was renamed,
   *                           if it exists.
   * @param  newEntry          The entry as it appears after it was renamed, if
   *                           it exists. .
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostOperationPluginResult doPostOperation(
              final ActiveOperationContext operationContext,
              final ModifyDNRequest request,
              final UpdatableModifyDNResult result, final Entry oldEntry,
              final Entry newEntry)
  {
    // No processing is performed by default.
    return PostOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary after all other processing
   * has been completed for a modify DN operation and the response has been
   * sent to the client.  This will be invoked only for modify DN operations
   * requested directly by clients, but not for modify DN operations received
   * from another server via replication.
   *
   * @param  operationContext  The context for the modify DN operation.
   * @param  request           The modify DN request that was processed.
   * @param  result            The result that was returned to the client.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostResponsePluginResult doPostResponse(
              final CompletedOperationContext operationContext,
              final ModifyDNRequest request, final ModifyDNResult result)
  {
    // No processing is performed by default.
    return PostResponsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary after all other processing
   * has been completed for a modify DN operation which has been received from
   * another server via replication.
   *
   * @param  operationContext  The context for the modify DN operation.
   * @param  request           The modify DN request that was processed.
   * @param  result            The result that was returned to the client.
   */
  public void doPostReplication(
                   final CompletedOperationContext operationContext,
                   final ModifyDNRequest request, final ModifyDNResult result)
  {
    // No processing is performed by default.
  }



  /**
   * Performs any processing which may be necessary before the server starts
   * processing for a search request.
   *
   * @param  operationContext  The context for the search operation.
   * @param  request           The search request to be processed.  It may be
   *                           altered if desired.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreParsePluginResult doPreParse(
              final ActiveSearchOperationContext operationContext,
              final UpdatableSearchRequest request,
              final UpdatableSearchResult result)
  {
    // No processing is performed by default.
    return PreParsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server actually
   * attempts to process the search in the backend.
   *
   * @param  operationContext  The context for the search operation.
   * @param  request           The search request to be processed.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreOperationPluginResult doPreOperation(
              final ActiveSearchOperationContext operationContext,
              final SearchRequest request, final UpdatableSearchResult result)
  {
    // No processing is performed by default.
    return PreOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server sends a
   * search result entry to the client.
   *
   * @param  operationContext  The context for the search operation.
   * @param  request           The search request being processed.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   * @param  entry             The entry to be returned to the client.  It may
   *                           be altered if desired.
   * @param  controls          The set of controls to be included with the
   *                           entry.  It may be altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public SearchEntryPluginResult doSearchEntry(
              final ActiveSearchOperationContext operationContext,
              final SearchRequest request, final UpdatableSearchResult result,
              final UpdatableEntry entry, final List<Control> controls)
  {
    // No processing is performed by default.
    return SearchEntryPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server sends a
   * search result reference to the client.
   *
   * @param  operationContext  The context for the search operation.
   * @param  request           The search request being processed.
   * @param  result            The result that will be returned to the client if
   *                           the plugin result indicates that processing on
   *                           the operation should be interrupted.  It may be
   *                           altered if desired.
   * @param  referralURLs      The set of referral URLs to be returned to the
   *                           client.  It may be altered if desired.
   * @param  controls          The set of controls to be included with the
   *                           reference.  It may be altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public SearchReferencePluginResult doSearchReference(
              final ActiveSearchOperationContext operationContext,
              final SearchRequest request, final UpdatableSearchResult result,
              final List<String> referralURLs, final List<Control> controls)
  {
    // No processing is performed by default.
    return SearchReferencePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server sends a
   * response for a search operation.
   *
   * @param  operationContext  The context for the search operation.
   * @param  request           The search request that was processed.
   * @param  result            The result to be returned to the client.  It may
   *                           be altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostOperationPluginResult doPostOperation(
              final ActiveSearchOperationContext operationContext,
              final SearchRequest request, final UpdatableSearchResult result)
  {
    // No processing is performed by default.
    return PostOperationPluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary after all other processing
   * has been completed for a search operation and the response has been sent to
   * the client.
   *
   * @param  operationContext  The context for the search operation.
   * @param  request           The search request that was processed.
   * @param  result            The result that was returned to the client.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PostResponsePluginResult doPostResponse(
              final CompletedSearchOperationContext operationContext,
              final SearchRequest request, final SearchResult result)
  {
    // No processing is performed by default.
    return PostResponsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server starts
   * processing for an unbind request.
   *
   * @param  operationContext  The context for the unbind operation.
   * @param  request           The unbind request to be processed.  It may be
   *                           altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public PreParsePluginResult doPreParse(
              final ActiveOperationContext operationContext,
              final UpdatableUnbindRequest request)
  {
    // No processing is performed by default.
    return PreParsePluginResult.SUCCESS;
  }



  /**
   * Performs any processing which may be necessary before the server sends an
   * intermediate response to the client.
   *
   * @param  operationContext      The context for the associated operation.
   * @param  result                The result that will be returned to the
   *                               client if the plugin result indicates that
   *                               processing on the operation should be
   *                               interrupted.  It may be altered if desired.
   * @param  intermediateResponse  The intermediate response to be returned to
   *                               the client.  It may be altered if desired.
   *
   * @return  Information about the result of the plugin processing.
   */
  public IntermediateResponsePluginResult doIntermediateResponse(
              final ActiveOperationContext operationContext,
              final UpdatableGenericResult result,
              final IntermediateResponse intermediateResponse)
  {
    // No processing is performed by default.
    return IntermediateResponsePluginResult.SUCCESS;
  }



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
